(* ::Package:: *)

BeginPackage["AVOfunctions`"]

(*This package contains functions to read in an earth model and plot it and to compute rays to specified receivers and plot them.   Additional functions are to compute the amplitude and phase of plane-wave arrivals at regular increments of offset and to plot arrival sections for AVO (amplitude-versus-offset) analysis.*)

getmodel::usage = "getmodel[modelfile]
where modelfile is a filename containing an earth model.  The model file is several rows of an ASCII file, with every row representing a layer and having 4 fields:
	1) layer thickness (km)
	2) layer P velocity (km/s)
	3) layer S velocity (km/s)
	4) layer density (gm/cc)
getmodel returns a list of lists where each sublist represents a layer, and the elements within each sublist are: 
	1) layer thickness (km)
	2) top depths of layers (km - negative values)
	3) bottom depths of layers (km -- negative values)
	4) layer P velocity (km/s)
	5) layer S velocity (km/s)
	6) layer density (gm/cc)";

plotmodel::usage = "plotmodel[model,parameter,lastlayer,plotlabel,maxoffset,exag]
where 
	model = the output of getmodel
	parameter = 'pvel', 'svel', or 'dens' to indicate which model parameter to plot
	lastlayer = number of last (deepest) layer to plot
	plotlabel = an ASCII label to put on top of plot
	maxoffset = maximum horizontal dimension (km)
	exag = vertical exaggeration -- model will be stretched vertically by this factor
plotmodel returns a graphic object.";

rayparms::usage = "rayparms[model,wave,no,maxoffset,lastlayer]
where
	model = the output of getmodel
	wave = 'SS' or 'PP' or 'PS' or 'SP'
	no = number of points (regularly spaced) from 0 to maxoffset (offset increment = maxoffset/(no-1)
	maxoffset = maximum horizontal distance from source to receiver (km)
	lastlayer = layer number at which wave will reflect off bottom
rayparms returns a list of lists, each sublist being {offset, p} where p is the ray parameter for that offset.";

plotrays::usage = "plotrays[model,wave,lastlayer,plotlabel,maxoffset,exag,pvso]
where
	model = the output of getmodel
    wave = 'SS' or 'PP' or 'PS' or 'SP'
    lastlayer = layer number at which wave will reflect off bottom
    plotlabel = an ASCII label to put on top of plot
    maxoffset = maximum horizontal distance from source to receiver (km)
    exag = vertical exaggeration -- model will be stretched vertically by this factor
	pvso = a list of lists, each level 1 element being the {offset (km), p (s/km)} pair defining a ray from source to receiver
plotrays returns a graphic object.";

plotSHAVO::usage = "plotSHAVO[model,lastlayer,plotlabel,maxoffset,pvso]
where
    model = the output of getmodel
    lastlayer = layer number at which wave will reflect off bottom
    plotlabel = an ASCII label to put on top of plot
    maxoffset = maximum horizontal distance from source to receiver (km)
	pvso = a list of lists, each level 1 element being the {offset (km), p (s/km)} pair defining a ray from source to receiver
plotSHAVO returns a graphic object.";

SHamppha::usage = "SHamppha[model,lastlayer,pvso]
where 
	model = the output of getmodel
	lastlayer = layer number at which wave will reflect off bottom
	pvso = returned list from rayparms
SHamppha returns a list of lists, each sublist representing an offset, from 0 to maxoffset, and each sublist containing 5 elements:
	1) offset (km), 
	2) takeoff angle (degrees), 
	3) total amplitude response, 
	4) total phase response (radians), 
	5) two-way travel time (s)";

plotPSVAVO::usage = "plotPSVAVO[model,wave,lastlayer,plotlabel,maxoffset,pvso]
where
    model = the output of getmodel
    wave = 'SS' or 'PP' or 'PS' or 'SP'
    lastlayer = layer number at which wave will reflect off bottom
    plotlabel = an ASCII label to put on top of plot
    maxoffset = maximum horizontal distance from source to receiver (km)
	pvso = a list of lists, each level 1 element being the {offset (km), p (s/km)} pair defining a ray from source to receiver
plotPSVAVO returns a graphic object.";

PSVamppha::usage = "PSVamppha[model,wave,lastlayer,pvso]
where 
	model = the output of getmodel
	wave = 'PP' or 'SS' or 'PS' or 'SP'
	lastlayer = layer number at which wave will reflect off bottom
	pvso = returned list from rayparms
PSVamppha returns a list of lists, each sublist representing an offset, from 0 to maxoffset, and each sublist containing the same as 5 elements as for SHamppha.";

plotarrivals::usage = "plotarrivals[model,system,wave,component,lastlayer,pvso,ascale,plotmode,t0,twave,dt,plotlabel,aspect]
where
    model = the output of getmodel
    system = 1 for SH or 2 for P-SV
    wave = 'PP' or 'SS' or 'PS' or 'SP' for P-SV system or 'SH' for SH system
    component = 'R' for radial or 'Z' for vertical (applies only to system = 2)
	lastlayer = layer number at which wave will reflect off bottom
	pvso = a list of lists, each level 1 element being the {offset (km), p (s/km)} pair defining a ray from source to receiver
    ascale = scaling factor on wavelet height
	plotmode = 'relative' for viewing amplitudes in correct relation to one another or
	           'uniform' for constant amplitude
	t0 = time scaling constant of Ricker wavelet (s)
	twave = Ricker wavelet span (s) from negative to positive cutoff
	dt = sampling interval for time series (s)
	plotlabel = label for the plot
	aspect = ratio of plot's vertical axis (offset) over horizontal axis (time)
plotarrivals returns a graphic object.";	


Needs["PlotLegends`"];


Begin["`Private`"]

(*Define a line width between Thin and Thick.*)
regular = AbsoluteThickness[1];


getmodel[modelfile_] :=Block[{alist, thick, bot, top, pvel, svel, dens},
alist = ReadList[modelfile,{Number,Number,Number,Number}];
If[Length[alist]==1,Print["model must be more than 1 layer"];Abort[]];
thick = alist[[All,1]];
bot = Accumulate[-thick];
top = bot + thick;
pvel = alist[[All,2]];
svel = alist[[All,3]];
dens = alist[[All,4]];
model = Transpose[{thick,top,bot,pvel,svel,dens}];
model
];


plotmodel[model_,parameter_,lastlayer_,plotlabel_,maxoffset_,exag_] := Block[{j, maxdepth, top, bot, ip, parmtype, parm, max, min, colors, temp},
If[lastlayer==1,Print["Cannot plot a 1-layer model."];Abort[]];
Which[
  parameter == "pvel", ip=4; parmtype="compressional velocity (km/s)",
  parameter == "svel", ip=5; parmtype="shear velocity (km/s)",
  parameter == "dens", ip=6; parmtype="density (gm/cc)"
];
maxdepth=model[[lastlayer,3]];
top=model[[All,2]];
bot=model[[All,3]];
parm=model[[All,ip]];
max=Max[parm[[1;;lastlayer]]];
min=Min[parm[[1;;lastlayer]]];
colors = "TemperatureMap";
temp=Show[Table[Graphics[{EdgeForm[Thin],ColorData[colors][(max-parm[[j]])/(max-min)],
Rectangle[{0,bot[[j]]},{maxoffset,top[[j]]}]}],{j,1,lastlayer}],
Frame->True,FrameLabel->{"offset (km)","depth (km)"},BaseStyle->{Medium},
PlotRangePadding->None,AspectRatio->(-maxdepth/maxoffset)*exag,
PlotLabel->Style[plotlabel<>"\n"<>parmtype<>"\n"<>
"minimum = "<>ToString[min]<>", maximum = "<>ToString[max],16],ImageSize->500];
ShowLegend[temp,{ColorData[colors][#1]&,10,"max","min",LegendShadow->None,LegendPosition->{1.1,-0.4}}]
];


plotrays[model_,wave_,lastlayer_,plotlabel_,maxoffset_,exag_,pvso_] := Block[{i, j, no, maxdepth, thick, top, bot, plot1, plot2, veld, velu, npts, rays, offset, p, dist, sinang, cosang, tanang},
no=Length[pvso];
maxdepth=model[[lastlayer,3]];
thick=model[[All,1]];
top=model[[All,2]];
bot=model[[All,3]];
(*Plot the model first.*)
plot1=Show[Table[Graphics[{EdgeForm[Thin],Opacity[0],
Rectangle[{0,bot[[j]]},{maxoffset,top[[j]]}]}],{j,1,lastlayer}],
Frame->True,FrameLabel->{"offset (km)","depth (km)"},BaseStyle->{Medium},
PlotRangePadding->None,AspectRatio->(-maxdepth/maxoffset)*exag,
PlotLabel->Style[plotlabel<>"\n"<>"vertical exaggeration = "<>ToString[exag],Medium],ImageSize->450];
If[wave=="PP" || wave=="PS",veld=model[[All,4]]];
If[wave=="SS" || wave=="SP",veld=model[[All,5]]];
If[wave=="PP" || wave=="SP",velu=model[[All,4]]];
If[wave=="SS" || wave=="PS",velu=model[[All,5]]];
npts=2*lastlayer+1;
(*Initialize the raypath table.*)
rays = Table[{0,0},{i,1,no},{j,1,npts}]; 
Do[
  offset=pvso[[i,1]];
  p=pvso[[i,2]];
  (*add last raypath point at given offset and depth zero*)
  rays[[i,npts]]={offset,0};
  (*Trace downgoing ray.*)
  dist=0.0;
  Do[
    sinang = veld[[j]]*p;
    cosang = Sqrt[1.0 - sinang^2];
    tanang = sinang/cosang;
    dist = dist + thick[[j]]*tanang;
    rays[[i,j+1]]={dist,bot[[j]]},
    {j,1,lastlayer}
  ];
  (*Trace upgoing ray.*)
  dist=0.0;
  Do[
    sinang = velu[[j]]*p;
    cosang = Sqrt[1.0 - sinang^2];
    tanang = sinang/cosang;
    dist = dist + thick[[j]]*tanang;
    rays[[i,npts-j]]={offset - dist,bot[[j]]},
    {j,1,lastlayer-1}
  ],
  {i,1,no}
];
plot2=ListPlot[rays,Joined->True,PlotStyle->Black];
Show[plot1,plot2]
];


rayparms[model_,wave_,no_,maxoffset_,lastlayer_] := Block[{i, j, thick, top, bot, pvel, svel, offsets, offset, p, pvsotemp, vel, avel, pmax, nrays, dp, dist1, dist2, sinang, cosang, tanang, pvso, startp, temp, ap, aval, ttemp},
If[lastlayer==1,Print["Cannot treat a 1-layer model."];Abort[]];
If[lastlayer>=Length[model],Print["reflecting layer must be < last layer (halfspace)"];Abort[]];
thick = model[[All,1]];
top = model[[All,2]];
bot = model[[All,3]];
pvel = model[[All,4]];
svel = model[[All,5]];
offsets=Table[(i-1)*maxoffset/(no-1),{i,1,no}];
(*Make a preliminary table of {offset,p} values to maxoffset by just running
rays until the maxoffset is exceeded. "p" is the vertical slowness and equals 
sin (\[Gamma])/c where \[Gamma] is the angle of the normal to the wavefront from vertical and 
c is the velocity.  A wave propagating vertically (\[Gamma] = 0) has zero slowness.  
The preliminary table will then be interpolated to exact offsets.*)
offset = 0.0;
p = 0.0;
pvsotemp={{offset,p}};
If[wave=="PP" || wave=="PS", vel = pvel];
If[wave=="SS" || wave=="SP", vel = svel];
(*Set the maximum p with the P velocity regardless of wave.*)
pmax = 1.0/pvel[[lastlayer+1]] - 0.001;
nrays = 1000; (*arbitrary number to sample ray parameter space sufficiently*)
dp = pmax/nrays;
While[offset < maxoffset,
  p = p + dp;
  dist1 = 0.0;
  dist2 = 0.0;
  (*Get ray horizontal distance for this p, looking at both sides of reflection point.*)
Label[begin]; 
 Do[
    If[wave=="PP" || wave=="PS",avel=pvel[[j]]];
    If[wave=="SS" || wave=="SP",avel=svel[[j]]];
    sinang = avel*p;
    cosang = Sqrt[1.0 - sinang^2];
    If[Head[cosang]===Complex, dp = dp/2; p = p - dp; Goto[begin]];
    tanang = sinang/cosang;
    dist1 = dist1 + thick[[j]]*tanang;
    If[wave=="PP" || wave=="SP",avel=pvel[[j]]];
    If[wave=="SS" || wave=="PS",avel=svel[[j]]];
    sinang = avel*p;
    cosang = Sqrt[1.0 - sinang^2];
    If[Head[cosang]===Complex, dp = dp/2; p = p - dp; Goto[begin]];
    tanang = sinang/cosang;
    dist2 = dist2 + thick[[j]]*tanang,
    {j,1,lastlayer}
  ];
  offset = dist1 + dist2;
  AppendTo[pvsotemp,{offset,p}]
];
(*Now find the p values associated with all exact offsets for the 
given reflection depth.  This uses the fast FindRoot function. The 
starting p value for each offset is obtained by interpolation of the 
table created above.*)
(*Set the zero-offset values and solve for the rest.*)
pvso = {{0.0,0.0}};
Do[
  Clear[p];
  offset=offsets[[i]];
  startp = Interpolation[pvsotemp,offset];
  If[wave=="PP" || wave=="SS", temp = Table[2*thick[[j]]*vel[[j]]/Sqrt[1.-p^2*vel[[j]]^2],{j,1,lastlayer}]];
  If[wave=="PS" || wave=="SP", temp = Table[thick[[j]]*(pvel[[j]]/Sqrt[1.-p^2*pvel[[j]]^2]+svel[[j]]/Sqrt[1.-p^2*svel[[j]]^2]),{j,1,lastlayer}]];
  ttemp= Total[temp];
  ap=FindRoot[p*ttemp == offset,{p,startp}];
  aval = Chop[p/.ap];
Print["offset,aval = ",offset," ",aval];
  AppendTo[pvso,{offset,aval}],
  {i,2,no}
];
(*Return the {offset,p} pairs for regular offsets, 0 to maxoffset.*)
pvso
];


plotSHAVO[model_,lastlayer_,plotlabel_,maxoffset_,pvso_] := Block[{no, data, offsets, angles, ampl, phas, times, i, k, avo, pvo, tvo, evo, maxt, maxa, plot1, plot2, plot3, plot4,space},
(*Compute the amplitude and phase for each ray.*)
no=Length[pvso];
data=SHamppha[model,lastlayer,pvso];
(*Plot the AVO data at the specified offsets.*)
offsets = data[[All,1]];
angles = data[[All,2]];
ampl = data[[All,3]];
phas = data[[All,4]];
times = data[[All,5]];
(*Need to unwind the phase.*)
Do[
  If[phas[[i]]\[LessSlantEqual]-\[Pi],phas[[i]]=phas[[i]]+2\[Pi]];
  If[phas[[i]]>\[Pi],phas[[i]]=phas[[i]]-2\[Pi]],
  {i,1,no}
];
avo = Transpose[{offsets,ampl}];
pvo = Transpose[{offsets,phas}];
tvo = Transpose[{offsets,times}];
evo = Transpose[{offsets,angles}];
maxt = (1+Floor[Max[times]/100.])*100; (*adjusts time extent to nearest 100 seconds*)
maxa = Ceiling[Max[ampl]]; (*adjusts amplitude extent to nearest greater integer*)
plot1=ListPlot[avo,PlotRange->{{0,maxoffset},{0,1}},Joined->True,PlotStyle->{regular,Black},Frame->True,FrameTicks->{{Table[k 0.25,{k,1,3}],None},{Automatic,None}},FrameLabel->{"offset (km)","amplitude",None,None},BaseStyle->Medium, AspectRatio->0.7,ImageSize->200];
plot2=ListPlot[pvo,PlotRange->{{0,maxoffset},{-\[Pi],\[Pi]}},Joined->True,PlotStyle->{regular,Black},Frame->True,FrameTicks->{{Table[k \[Pi]/2,{k,-2,2}],None},{Automatic,None}},FrameLabel->{"offset (km)","phase (radians)",None,None},BaseStyle->Medium, AspectRatio->0.7,ImageSize->200];
plot3=ListPlot[tvo,PlotRange->{{0,maxoffset},{0,200}},Joined->True,PlotStyle->{regular,Black},Frame->True,FrameTicks->{{Table[k (200/5),{k,0,5}],None},{Automatic,None}},FrameLabel->{"offset (km)","travel time (s)",None,None},BaseStyle->Medium, AspectRatio->0.7,ImageSize->200];
plot4=ListPlot[evo,PlotRange->{{0,maxoffset},{0,90}},Joined->True,PlotStyle->{regular,Black},Frame->True,FrameTicks->{{Table[k 15,{k,0,6}],None},{Automatic,None}},FrameLabel->{"offset (km)","takeoff angle \n(degrees)",None,None},BaseStyle->Medium, AspectRatio->0.7,ImageSize->200];
space=Graphics[{},ImageSize->{10,100}];
Column[{
Text@Style[plotlabel<>"SH reflection in layer "<>ToString[lastlayer],14],
Grid[{{plot1,space,plot2},{plot3,space,plot4}}]},
Alignment->Center
]
];


SHamppha[model_,lastlayer_,pvso_]:=Block[{i, j, coef, no, thick, vel, svel, dens, data, Ta, Ra, Tp, Rp, depth, dist, time, offset, sinang, cosang, \[Rho]1, \[Rho]2, \[Beta]1, \[Beta]2, p, \[Gamma]d, \[Gamma]u, \[Gamma], down, up, r, s, contrib, angle, amp, pha},
coef=coefSH[];
no=Length[pvso];
thick=model[[All,1]];
vel=model[[All,5]];
dens=model[[All,6]];
(*Initialize a table for results.*)
data={};
Do[
  Ta=1.0;
  Ra=1.0;
  Tp=0.0;
  Rp=0.0;
  (*Trace the ray to the reflecting boundary.*)
  depth = 0.0;
  dist = 0.0;
  time = 0.0;
  p = pvso[[i,2]];
  Do[
    \[Gamma]d=ArcSin[vel[[j]]*p]/Degree;
    \[Gamma]u=ArcSin[vel[[j+1]]*p]/Degree;
    \[Rho]1 = dens[[j]];
    \[Rho]2 = dens[[j+1]];
    \[Beta]1 = vel[[j]];
    \[Beta]2 = vel[[j+1]];
    If[j<lastlayer,
      (*Get up-down transmission coef. product.*)
      down = coef[[1,2]]/.\[Gamma]->\[Gamma]d/.r->\[Rho]2/\[Rho]1/.s->\[Beta]2/\[Beta]1;
      up = coef[[2,1]]/.\[Gamma]->\[Gamma]u/.r->\[Rho]2/\[Rho]1/.s->\[Beta]2/\[Beta]1;
      contrib=down*up;
      Ta=Ta*Abs[contrib]; 
      Tp=Tp+Arg[contrib]
    ];
    If[j==lastlayer,
      (*Get reflection coefficient.*)
      contrib=coef[[1,1]]/.\[Gamma]->\[Gamma]d/.r->\[Rho]2/\[Rho]1/.s->\[Beta]2/\[Beta]1;
      Ra=Abs[contrib];
      Rp=Arg[contrib]
    ];
    sinang = vel[[j]]*p;
    cosang = Sqrt[1.0 - sinang^2];
    time = time + (thick[[j]]/cosang)/vel[[j]],
    {j,1,lastlayer}
  ];
  offset = pvso[[i,1]];
  angle = ArcSin[vel[[1]]*p]/Degree; (*takeoff angle from vertical at surface*)
  amp = Ra*Ta;
  pha = Rp+Tp;
  time = 2*time;
  AppendTo[data,{offset,angle,amp,pha,time}],
  {i,1,no}
];
(*Return the data matrix, one row per offset.*)
data
];


plotPSVAVO[model_,wave_,lastlayer_,plotlabel_,maxoffset_,pvso_]:= Block[{data,offsets,angles,ampl,phas,times,i,k,no,avo,pvo,tvo,evo,avoz,avor,maxt,maxa,p1,p2,p3,p4,p5,p6,space},
(*Compute the amplitude and phase for each ray and plot them versus offset.*)
no=Length[pvso];
data=PSVamppha[model,wave,lastlayer,pvso];
(*Plot the AVO data at the specified offsets.*)
offsets = data[[All,1]];
angles = data[[All,2]];
ampl = data[[All,3]];
phas = data[[All,4]];
times = data[[All,5]];
(*Need to unwind the phase.*)
Do[
  If[phas[[i]]\[LessSlantEqual]-\[Pi],phas[[i]]=phas[[i]]+2\[Pi]];
  If[phas[[i]]>\[Pi],phas[[i]]=phas[[i]]-2\[Pi]],
  {i,1,no}
];
avo = Transpose[{offsets,ampl}];
pvo = Transpose[{offsets,phas}];
tvo = Transpose[{offsets,times}];
evo = Transpose[{offsets,angles}];
(*Compute the amplitude recorded on the vertical and radial components.*)
If[wave=="PP",avoz = Transpose[{offsets,ampl*Cos[angles \[Degree]]}];avor = Transpose[{offsets,ampl*Sin[angles \[Degree]]}]];
If[wave=="SS",avoz = Transpose[{offsets,ampl*Sin[angles \[Degree]]}];avor = Transpose[{offsets,ampl*Cos[angles \[Degree]]}]];
If[wave=="SP",avoz = Transpose[{offsets,ampl*Cos[angles \[Degree]]}];avor = Transpose[{offsets,ampl*Sin[angles \[Degree]]}]];
If[wave=="PS",avoz = Transpose[{offsets,ampl*Sin[angles \[Degree]]}];avor = Transpose[{offsets,ampl*Cos[angles \[Degree]]}]];
maxt = (1+Floor[Max[times]/100.])*100; (*adjusts time extent to nearest 100 seconds*)
maxa = Ceiling[Max[ampl]]; (*adjusts amplitude extent to nearest greater integer*)
p1=ListPlot[avo,PlotRange->{{0,maxoffset},{0,maxa}},Joined->True,PlotStyle->{regular,Black},Frame->True,FrameTicks->{{Table[k 0.25,{k,1,3}],None},{Automatic,None}},FrameLabel->{"offset (km)","amplitude",None,None},BaseStyle->Medium,AspectRatio->0.7,ImageSize->200];
p2=ListPlot[pvo,PlotRange->{{0,maxoffset},{-\[Pi],\[Pi]}},Joined->True,PlotStyle->{regular,Black},Frame->True,FrameTicks->{{Table[k \[Pi]/2,{k,-2,2}],None},{Automatic,None}},FrameLabel->{"offset (km)","phase (radians)",None,None},BaseStyle->Medium,AspectRatio->0.7,ImageSize->200];
p3=ListPlot[tvo,PlotRange->{{0,maxoffset},{0,maxt}},Joined->True,PlotStyle->{regular,Black},Frame->True,FrameTicks->{{Table[k (maxt/5),{k,0,5}],None},{Automatic,None}},FrameLabel->{"offset (km)","travel time (s)",None,None},BaseStyle->Medium,AspectRatio->0.7,ImageSize->200];
p4=ListPlot[evo,PlotRange->{{0,maxoffset},{0,90}},Joined->True,PlotStyle->{regular,Black},Frame->True,FrameTicks->{{Table[k 15,{k,0,6}],None},{Automatic,None}},FrameLabel->{"offset (km)","takeoff angle \n(degrees)",None,None},BaseStyle->Medium,AspectRatio->0.7,ImageSize->200];
p5=ListPlot[avoz,PlotRange->{{0,maxoffset},{0,maxa}},Joined->True,PlotStyle->{regular,Black},Frame->True,FrameTicks->{{Table[k 0.25,{k,1,3}],None},{Automatic,None}},FrameLabel->{"offset (km)","vertical amplitude",None,None},BaseStyle->Medium,AspectRatio->0.7,ImageSize->200];
p6=ListPlot[avor,PlotRange->{{0,maxoffset},{0,maxa}},Joined->True,PlotStyle->{regular,Black},Frame->True,FrameTicks->{{Table[k 0.25,{k,1,3}],None},{Automatic,None}},FrameLabel->{"offset (km)","radial amplitude",None,None},BaseStyle->Medium,AspectRatio->0.7,ImageSize->200];
space=Graphics[{},ImageSize->{10,100}];
Column[{
Text@Style[plotlabel<>wave<>" reflection in layer "<>ToString[lastlayer],14],
Grid[{{p1,space,p2},{p3,space,p4},{p5,space,p6}}]
},
Alignment->Center]
];



PSVamppha[model_,wave_,lastlayer_,pvso_]:=Block[{i, j, coef, coefd, coefu, coefr, no, thick, vel, pvel, svel, dens, data, Ta, Ra, Tp, Rp, depth, dist, time, timed, timeu, offset, sinang, cosang, \[Rho]1, \[Rho]2, \[Alpha]1, \[Alpha]2, \[Beta]1, \[Beta]2, p, \[Gamma]d, \[Gamma]u, \[Gamma], down, up, r, s, v, w, contrib, angle, amp, pha},
coef=coefPSV[];
no=Length[pvso];
thick=model[[All,1]];
pvel=model[[All,4]];
svel=model[[All,5]];
dens=model[[All,6]];
(*Initialize a table for results.*)
data={};
Do[
  Ta=1.0;
  Ra=1.0;
  Tp=0.0;
  Rp=0.0;
  (*Trace the ray to the reflecting boundary.*)
  depth = 0.0;
  dist = 0.0;
  timed = 0.0;
  timeu = 0.0;
  p = pvso[[i,2]];
  Do[
    If[wave=="PP",\[Gamma]d=ArcSin[pvel[[j]]*p]/Degree;\[Gamma]u=ArcSin[pvel[[j+1]]*p]/Degree];
    If[wave=="SS",\[Gamma]d=ArcSin[svel[[j]]*p]/Degree;\[Gamma]u=ArcSin[svel[[j+1]]*p]/Degree];
    If[wave=="PS",\[Gamma]d=ArcSin[pvel[[j]]*p]/Degree;\[Gamma]u=ArcSin[svel[[j+1]]*p]/Degree];
    If[wave=="SP",\[Gamma]d=ArcSin[svel[[j]]*p]/Degree;\[Gamma]u=ArcSin[pvel[[j+1]]*p]/Degree];
    \[Rho]1 = dens[[j]];
    \[Rho]2 = dens[[j+1]];
    \[Alpha]1 = pvel[[j]];
    \[Alpha]2 = pvel[[j+1]];
    \[Beta]1 = svel[[j]];
    \[Beta]2 = svel[[j+1]];
    If[wave=="PP",coefd=coef[[1,3]];coefu=coef[[3,1]]];
    If[wave=="SS",coefd=coef[[2,4]];coefu=coef[[4,2]]];
    If[wave=="PS",coefd=coef[[1,3]];coefu=coef[[4,2]]];
    If[wave=="SP",coefd=coef[[2,4]];coefu=coef[[3,1]]];
    If[j<lastlayer,
      (*Get up-down transmission coef. product.*)
      down = coefd/.\[Gamma]->\[Gamma]d/.r->\[Rho]2/\[Rho]1/.s->\[Beta]2/\[Beta]1/.v->\[Alpha]1/\[Beta]1/.w->\[Alpha]2/\[Beta]2;
      up   = coefu/.\[Gamma]->\[Gamma]u/.r->\[Rho]2/\[Rho]1/.s->\[Beta]2/\[Beta]1/.v->\[Alpha]1/\[Beta]1/.w->\[Alpha]2/\[Beta]2;
      contrib=down*up;
      Ta=Ta*Abs[contrib]; 
      Tp=Tp+Arg[contrib]
    ];
    If[j==lastlayer,
      (*Get reflection coefficient.*)
      If[wave=="PP",coefr=coef[[1,1]]];
      If[wave=="SS",coefr=coef[[2,2]]];
      If[wave=="PS",coefr=coef[[1,2]]];
      If[wave=="SP",coefr=coef[[2,1]]];
      contrib = coefr/.\[Gamma]->\[Gamma]d/.r->\[Rho]2/\[Rho]1/.s->\[Beta]2/\[Beta]1/.v->\[Alpha]1/\[Beta]1/.w->\[Alpha]2/\[Beta]2;
      Ra=Abs[contrib];
      Rp=Arg[contrib]
    ];
    If[wave=="PP" || wave=="PS", 
      sinang = pvel[[j]]*p;
      cosang = Sqrt[1.0 - sinang^2];
      timed = timed + (thick[[j]]/cosang)/pvel[[j]]
    ];
    If[wave=="SS" || wave=="SP", 
      sinang = svel[[j]]*p;
      cosang = Sqrt[1.0 - sinang^2];
      timed = timed + (thick[[j]]/cosang)/svel[[j]]
    ];
    If[wave=="PP" || wave=="SP", 
      sinang = pvel[[j]]*p;
      cosang = Sqrt[1.0 - sinang^2];
      timeu = timeu + (thick[[j]]/cosang)/pvel[[j]]
    ];
    If[wave=="SS" || wave=="PS", 
      sinang = svel[[j]]*p;
      cosang = Sqrt[1.0 - sinang^2];
      timeu = timeu + (thick[[j]]/cosang)/svel[[j]]
    ],
    {j,1,lastlayer}
  ];
  offset = pvso[[i,1]];
  (*Compute takeoff angles.*)
  If[wave=="PP" || wave=="PS", angle= ArcSin[pvel[[1]]*p]/Degree]; 
  If[wave=="SS" || wave=="SP", angle= ArcSin[svel[[1]]*p]/Degree]; 
  amp = Ra*Ta;
  pha = Rp+Tp;
  time = timed+timeu;
  AppendTo[data,{offset,angle,amp,pha,time}],
  {i,1,no}
];
(*Return the data matrix, one row per offset.*)
data
];


plotarrivals[model_,system_,wave_,component_,lastlayer_,pvso_,ascale_,plotmode_,t0_,twave_,dt_,plotlabel_,aspect_]:=Block[{data, i, k, no, offsets, maxoffset, ampl, phas, times, nt, rickerspec, tmax, tt, mt, zeros, plot, rickerspecp, rickerp, rickerplist, nz, nr, bscale, t, \[Tau], \[Omega]},
no = Length[pvso];
If[system==1, data=SHamppha[model,lastlayer,pvso]];
If[system==2, data=PSVamppha[model,wave,lastlayer,pvso]];
offsets = data[[All,1]];
maxoffset = offsets[[no]];
ampl = data[[All,3]];
phas = data[[All,4]];
times = data[[All,5]];
nt = 2*Round[twave/dt] + 1;
rickerspec=(2 E^(-(1/2) \[Tau]^2 \[Omega]^2) \[Tau]^(5/2) \[Omega]^2)/(Sqrt[3] \[Pi]^(1/4))/.\[Tau]->t0;
tmax = Max[times];
tmax = 10* (Ceiling[tmax/10]+1);
mt=tmax/dt + 1;
tt=Table[(i-1)*dt,{i,1,mt}];
zeros=Table[0,{i,1,mt}];
plot=Table[0,{i,1,no}];
Do[
  rickerspecp = Chop[rickerspec*E^(I*phas[[k]]*Sign[\[Omega]])];
  rickerp =Chop[InverseFourierTransform[rickerspecp,\[Omega],t]];
  rickerplist=Table[rickerp,{t,-twave,twave,dt}];
  (*Insert wavelet into trace at correct time.*)
  nz=Round[times[[k]]/dt] - (nt-1)/2;
  rickerplist= Join[Take[zeros,nz],rickerplist];
  nr=Length[rickerplist];
  rickerplist=Join[rickerplist,Take[zeros,mt-nr]];
  If[plotmode=="relative",bscale=ampl[[k]]*ascale,bscale =(1/Max[Abs[rickerplist]])*ascale];
  plot[[k]]=ListPlot[Transpose[{tt,bscale*rickerplist+offsets[[k]]}],PlotRange->{{0,tmax},All},PlotStyle->Black,Joined->True],
  {k,1,no}
];
Show[plot,PlotRange->{{0,tmax},{-10,maxoffset+10}},
Frame->True,FrameLabel->{"time (s)","offset (km)"},FrameStyle->regular,
BaseStyle->{Medium},PlotLabel->plotlabel,AspectRatio->aspect,ImageSize->450]
];


(*
SH coefficients:
This cell defines the reflection and transmission coefficients of plane SH waves in terms of layer parameter ratios: r for density ratio of layer 2 over layer 1 and s for shear velocity ratio of layer 2 over layer 1.  The expressions should be evaluated for specific incidence angles \[Gamma] (degrees) and for specific r and s.  The starting point are the expressions in Aki & Richards (eq. 5.33).  The waves are incident on the plane boundary of two elastic, homogeneous half spaces where subscripts 1 and 2 refer to the upper and lower media.  The notebook creates 4 expressions for calculation of reflection/transmission coefficients of the 4 scattered waves from the 2 possible incident waves (Sd, Su).  For instance, SdSu is the scattered Su from an incident Sd.  After making this set of definitions, they are returned in a matrix (list of lists) like:   

{
{SdSu,SdSd},
{SuSu,SuSd}
}
*)

coefSH[] := Block[{r, s, \[Beta]1, \[Beta]2, \[Rho]1, \[Rho]2, j1, j2, csj1, csj2, \[Gamma], \[CapitalDelta]},
         
         (*Set up substitutions such that final expressions are in terms of r, s.*)
         \[Rho]2 = r \[Rho]1;(*This eliminates both densities as \[Rho]1 eventually cancels everywhere.*)
         \[Beta]2 = s \[Beta]1;(*This eliminates both velocities as \[Beta]1 eventually cancels everywhere.*)
         
         \[CapitalDelta] = \[Rho]1 \[Beta]1 csj1 + \[Rho]2 \[Beta]2 csj2;
   
         (*Sd expressions*)
         csj1 = Sqrt[1 - Sin[j1]^2];
         csj2 = Sqrt[1 - (\[Beta]2/\[Beta]1)^2 Sin[j1]^2];
         
         SdSu = (\[Rho]1 \[Beta]1 csj1 - \[Rho]2 \[Beta]2 csj2)/\[CapitalDelta];
         SdSu = Cancel[SdSu];
         SdSu = SdSu /. j1 -> \[Gamma] \[Degree];
         
         SdSd = (2 \[Rho]1 \[Beta]1 csj1)/\[CapitalDelta];
         SdSd = Cancel[SdSd];
         SdSd = SdSd /. j1 -> \[Gamma] \[Degree];
         
         (*Su expressions*)
         csj1 = Sqrt[1 - (\[Beta]1/\[Beta]2)^2 Sin[j2]^2];
         csj2 = Sqrt[1 - Sin[j2]^2];
               
         SuSu = (2 \[Rho]2 \[Beta]2 csj2)/\[CapitalDelta];
         SuSu = Cancel[SuSu];
         SuSu = SuSu /. j2 -> \[Gamma] \[Degree];
         
         SuSd = (-\[Rho]1 \[Beta]1 csj1 + \[Rho]2 \[Beta]2 csj2)/\[CapitalDelta];
         SuSd = Cancel[SuSd];
         SuSd = SuSd /. j2 -> \[Gamma] \[Degree];
         
         (*Return the 2x2 matrix of coefficients.*)
         {{SdSu, SdSd}, {SuSu, SuSd}}
         ];



(*
P-SV coefficients
This cell expresses the reflection and transmission coefficients of plane P-SV waves in terms of layer parameter ratios: r for density ratio of layer 2 over layer 1,  s for shear velocity ratio of layer 2 over layer 1, v for ratio of compressional/shear velocity in the top layer, and w for similar ratio in the bottom layer.  The expressions should be evaluated for specific incidence angles \[Gamma] (degrees) and for specific v, w, r, s.  The starting point are the expressions in Aki & Richards (eq. 5.40).  The waves are incident on the plane boundary of two homogeneous half-spaces where subscripts 1 and 2 refer to the upper and lower media.  The notebook creates sixteen expressions for calculation of reflection/transmission coefficients of the sixteen scattered waves from the 4 possible incident waves (Pd,Sd,Pu,Su).  For instance, PdPu is the scattered Pu from an incident Pd.  After making this set of definitions, they are returned in a matrix (list of lists) like:
{
{PdPu,PdSu,PdPd,PdSd},
{SdPu,SdSu,SdPd,SdSd},
{PuPu,PuSu,PuPd,PuSd},
{SuPu,SuSu,SuPd,SuSd}
}
*)

coefPSV[] := Block[{r, s, v, w, \[Alpha]1, \[Alpha]2, \[Beta]1, \[Beta]2, \[Rho]1, \[Rho]2, i1, i2, j1, j2, csi1, csi2, csj1, csj2, a, b, c, d, e, f, g, h, dd, p, factor, \[Gamma]},
         
         (*Set up substitutions such that final expressions are in terms of v, w, r, s.*)
         \[Alpha]1 = \[Beta]1 v;(*This eliminates \[Alpha]1 from the equations.*)
         \[Alpha]2 = \[Beta]2 w;(*This eliminates \[Alpha]2 from the equations.*)
         \[Rho]2 = r \[Rho]1;(*This eliminates both densities as \[Rho]1 eventually cancels everywhere.*)
         \[Beta]2 = s \[Beta]1;(*This eliminates both velocities as \[Beta]1 eventually cancels everywhere.*)
         
         (*Pd expressions*)
         csi1 = Sqrt[1 - Sin[i1]^2];
         csj1 = Sqrt[1 - (1/v^2) Sin[i1]^2];
         csi2 = Sqrt[1 - (w^2/v^2) s^2 Sin[i1]^2];
         csj2 = Sqrt[1 - (1/v^2) s^2 Sin[i1]^2];
         a = \[Rho]2 (1 - 2 \[Beta]2^2 p^2) - \[Rho]1 (1 - 2 \[Beta]1^2 p^2);
         b = \[Rho]2 (1 - 2 \[Beta]2^2 p^2) + 2 \[Rho]1 \[Beta]1^2 p^2;
         c = \[Rho]1 (1 - 2 \[Beta]1^2 p^2) + 2 \[Rho]2 \[Beta]2^2 p^2;
         d = 2 (\[Rho]2 \[Beta]2^2 - \[Rho]1 \[Beta]1^2);
         e = b csi1/\[Alpha]1 + c csi2/\[Alpha]2;
         f = b csj1/\[Beta]1 + c csj2/\[Beta]2;
         g = a - d csi1 csj2/(\[Alpha]1 \[Beta]2);
         h = a - d csi2 csj1/(\[Alpha]2 \[Beta]1);
         dd = e f + g h p^2;
         
         PdPu = ((b csi1/\[Alpha]1 - c csi2/\[Alpha]2) f - h p^2 (a + d csi1 csj2/(\[Alpha]1 \[Beta]2)))/dd;
         PdPu = Cancel[PdPu];
         PdPu = PdPu /. p -> Sin[i1]/\[Alpha]1;
         PdPu = Cancel[PdPu];
         PdPu = PdPu /. i1 -> \[Gamma] \[Degree];
         
         PdSu = -2 csi1 (a b + c d csi2 csj2/(\[Alpha]2 \[Beta]2)) p/(\[Beta]1 dd);
         PdSu = Cancel[PdSu];
         PdSu = PdSu /. p -> Sin[i1]/\[Alpha]1;
         PdSu = Cancel[PdSu];
         PdSu = PdSu /. i1 -> \[Gamma] \[Degree];
         
         PdPd = 2 \[Rho]1 csi1 f/(\[Alpha]2 dd);
         PdPd = Cancel[PdPd];
         PdPd = PdPd /. p -> Sin[i1]/\[Alpha]1;
         PdPd = Cancel[PdPd];
         PdPd = PdPd /. i1 -> \[Gamma] \[Degree];
         
         PdSd = 2 \[Rho]1 csi1 h p/(\[Beta]2 dd);
         PdSd = Cancel[PdSd];
         PdSd = PdSd /. p -> Sin[i1]/\[Alpha]1;
         PdSd = Cancel[PdSd];
         PdSd = PdSd /. i1 -> \[Gamma] \[Degree];
         
         (*Sd expressions*)
         csi1 = Sqrt[1 - v^2 Sin[j1]^2];
         csj1 = Sqrt[1 - Sin[j1]^2];
         csi2 = Sqrt[1 - (w s)^2 Sin[j1]^2];
         csj2 = Sqrt[1 - s^2 Sin[j1]^2];
         a = \[Rho]2 (1 - 2 \[Beta]2^2 p^2) - \[Rho]1 (1 - 2 \[Beta]1^2 p^2);
         b = \[Rho]2 (1 - 2 \[Beta]2^2 p^2) + 2 \[Rho]1 \[Beta]1^2 p^2;
         c = \[Rho]1 (1 - 2 \[Beta]1^2 p^2) + 2 \[Rho]2 \[Beta]2^2 p^2;
         d = 2 (\[Rho]2 \[Beta]2^2 - \[Rho]1 \[Beta]1^2);
         e = b csi1/\[Alpha]1 + c csi2/\[Alpha]2;
         f = b csj1/\[Beta]1 + c csj2/\[Beta]2;
         g = a - d csi1 csj2/(\[Alpha]1 \[Beta]2);
         h = a - d csi2 csj1/(\[Alpha]2 \[Beta]1);
         dd = e f + g h p^2;
         
         SdPu = -2 csj1 (a b + c d csi2 csj2/(\[Alpha]2 \[Beta]2)) p/(\[Alpha]1 dd);
         SdPu = Cancel[SdPu];
         SdPu = SdPu /. p -> Sin[j1]/\[Beta]1;
         SdPu = Cancel[SdPu];
         SdPu = SdPu /. j1 -> \[Gamma] \[Degree];
         
         SdSu = ((-b csj1/\[Beta]1 + c csj2/\[Beta]2) e + g p^2 (a + d csi2 csj1/(\[Alpha]2 \[Beta]1)))/dd;
         SdSu = Cancel[SdSu];
         SdSu = SdSu /. p -> Sin[j1]/\[Beta]1;
         SdSu = Cancel[SdSu];
         SdSu = SdSu /. j1 -> \[Gamma] \[Degree];
         
         SdPd = -2 \[Rho]1 csj1 g p/(\[Alpha]2 dd);
         SdPd = Cancel[SdPd];
         SdPd = SdPd /. p -> Sin[j1]/\[Beta]1;
         SdPd = Cancel[SdPd];
         SdPd = SdPd /. j1 -> \[Gamma] \[Degree];
         
         SdSd = 2 \[Rho]1 csj1 e/(\[Beta]2 dd);
         SdSd = Cancel[SdSd];
         SdSd = SdSd /. p -> Sin[j1]/\[Beta]1;
         SdSd = Cancel[SdSd];
         SdSd = SdSd /. j1 -> \[Gamma] \[Degree];
         
         (*Pu expressions*)
         csi1 = Sqrt[1 - (v/(w s))^2 Sin[i2]^2];
         csj1 = Sqrt[1 - (1/(w s))^2 Sin[i2]^2];
         csi2 = Sqrt[1 - Sin[i2]^2];
         csj2 = Sqrt[1 - (1/w)^2 Sin[i2]^2];
         a = \[Rho]2 (1 - 2 \[Beta]2^2 p^2) - \[Rho]1 (1 - 2 \[Beta]1^2 p^2);
         b = \[Rho]2 (1 - 2 \[Beta]2^2 p^2) + 2 \[Rho]1 \[Beta]1^2 p^2;
         c = \[Rho]1 (1 - 2 \[Beta]1^2 p^2) + 2 \[Rho]2 \[Beta]2^2 p^2;
         d = 2 (\[Rho]2 \[Beta]2^2 - \[Rho]1 \[Beta]1^2);
         e = b csi1/\[Alpha]1 + c csi2/\[Alpha]2;
         f = b csj1/\[Beta]1 + c csj2/\[Beta]2;
         g = a - d csi1 csj2/(\[Alpha]1 \[Beta]2);
         h = a - d csi2 csj1/(\[Alpha]2 \[Beta]1);
         dd = e f + g h p^2;
         
         PuPu = 2 \[Rho]2 csi2 f/(\[Alpha]1 dd);
         PuPu = Cancel[PuPu];
         PuPu = PuPu /. p -> Sin[i2]/\[Alpha]2;
         PuPu = Cancel[PuPu];
         PuPu = PuPu /. i2 -> \[Gamma] \[Degree];
         
         PuSu = -2 \[Rho]2 csi2 g p/(\[Beta]1 dd);
         PuSu = Cancel[PuSu];
         PuSu = PuSu /. p -> Sin[i2]/\[Alpha]2;
         PuSu = Cancel[PuSu];
         PuSu = PuSu /. i2 -> \[Gamma] \[Degree];
         
         PuPd = ((-b csi1/\[Alpha]1 + c csi2/\[Alpha]2) f - g p^2 (a + d csi2 csj1/(\[Alpha]2 \[Beta]1)))/dd;
         PuPd = Cancel[PuPd];
         PuPd = PuPd /. p -> Sin[i2]/\[Alpha]2;
         PuPd = Cancel[PuPd];
         PuPd = PuPd /. i2 -> \[Gamma] \[Degree];
         
         PuSd = 2 csi2 (a c + b d csi1 csj1/(\[Alpha]1 \[Beta]1)) p/(\[Beta]2 dd);
         PuSd = Cancel[PuSd];
         PuSd = PuSd /. p -> Sin[i2]/\[Alpha]2;
         PuSd = Cancel[PuSd];
         PuSd = PuSd /. i2 -> \[Gamma] \[Degree];
         
         (*Su expressions*)
         csi1 = Sqrt[1 - (v/s)^2 Sin[j2]^2];
         csj1 = Sqrt[1 - (1/s)^2 Sin[j2]^2];
         csi2 = Sqrt[1 - w^2 Sin[j2]^2];
         csj2 = Sqrt[1 - Sin[j2]^2];
         a = \[Rho]2 (1 - 2 \[Beta]2^2 p^2) - \[Rho]1 (1 - 2 \[Beta]1^2 p^2);
         b = \[Rho]2 (1 - 2 \[Beta]2^2 p^2) + 2 \[Rho]1 \[Beta]1^2 p^2;
         c = \[Rho]1 (1 - 2 \[Beta]1^2 p^2) + 2 \[Rho]2 \[Beta]2^2 p^2;
         d = 2 (\[Rho]2 \[Beta]2^2 - \[Rho]1 \[Beta]1^2);
         e = b csi1/\[Alpha]1 + c csi2/\[Alpha]2;
         f = b csj1/\[Beta]1 + c csj2/\[Beta]2;
         g = a - d csi1 csj2/(\[Alpha]1 \[Beta]2);
         h = a - d csi2 csj1/(\[Alpha]2 \[Beta]1);
         dd = e f + g h p^2;
         
         SuPu = 2 \[Rho]2 csj2 h p/(\[Alpha]1 dd);
         SuPu = Cancel[SuPu];
         SuPu = SuPu /. p -> Sin[j2]/\[Beta]2;
         SuPu = Cancel[SuPu];
         SuPu = SuPu /. j2 -> \[Gamma] \[Degree];
         
         SuSu = 2 \[Rho]2 csj2 e/(\[Beta]1 dd);
         SuSu = Cancel[SuSu];
         SuSu = SuSu /. p -> Sin[j2]/\[Beta]2;
         SuSu = Cancel[SuSu];
         SuSu = SuSu /. j2 -> \[Gamma] \[Degree];
         
         SuPd = 2 csj2 (a c + b d csi1 csj1/(\[Alpha]1 \[Beta]1)) p/(\[Alpha]2 dd);
         SuPd = Cancel[SuPd];
         SuPd = SuPd /. p -> Sin[j2]/\[Beta]2;
         SuPd = Cancel[SuPd];
         SuPd = SuPd /. j2 -> \[Gamma] \[Degree];
         
         SuSd = ((b csj1/\[Beta]1 - c csj2/\[Beta]2) e + h p^2 (a + d csi1 csj2/(\[Alpha]1 \[Beta]2)))/dd;
         SuSd = Cancel[SuSd];
         SuSd = SuSd /. p -> Sin[j2]/\[Beta]2;
         SuSd = Cancel[SuSd];
         SuSd = SuSd /. j2 -> \[Gamma] \[Degree];
         
         (*Return the 4x4 matrix of coefficients.*)
         {{PdPu, PdSu, PdPd, PdSd}, {SdPu, SdSu, SdPd, SdSd}, {PuPu, PuSu, PuPd, PuSd}, {SuPu, SuSu, SuPd, SuSd}}
         ];



End[]
EndPackage[]
