(* ::Package:: *)

BeginPackage["RTfunctions`"]

(*This notebook contains several function assignments for computing reflection and transmission coefficients at the boundary between two elastic half spaces and also function assignments to plot the results.*)  

plotBrewster::usage = "plotBrewster[smin_Real, smax_Real]
where
	smin = minimum value of the ratio s to use on the horizontal axis
	smax = maximum value of the ratio s to use on the horizontal axis";

plotSHcoef::usage = "plotSHcoef[acode_String, ri_Real, si_Real]
where
	acode = 'Sd' or 'Su'
	ri = density ratio of the two layers (bottom to top)
	si = shear velocity ratio of the two layers (bottom to top)";

plotSHflux::usage = "plotSHflux[acode_String, ri_Real, si_Real]
where
	acode = 'Sd' or 'Su'
	ri = density ratio of the two layers (bottom to top)
	si = shear velocity ratio of the two layers (bottom to top)";

plotSHenergy::usage = "plotSHenergy[acode_String, ri_Real, smin_Real, smax_Real]
where
	acode = 'Sd' or 'Su'
	ri = density ratio of the two layers (bottom to top)
	smin = minimum value of the ratio s to use on the horizontal axis
	smax = maximum value of the ratio s to use on the horizontal axis";

plotPSVcoef::usage = "plotPSVcoef[acode_String, ri_Real, si_Real, vi_Real, wi_Real]
where
	acode = 'Pd' or 'Pu' or 'Sd' or 'Su'
	ri = density ratio of the two layers (bottom to top)
	si = shear velocity ratio of the two layers (bottom to top)
	vi = compressional-to-shear velocity ratio in top layer
	wi = compressional-to-shear velocity ratio in bottom layer";

plotPSVflux::usage = "plotPSVflux[acode_String, ri_Real, si_Real, vi_Real, wi_Real]
where
	acode = 'Pd' or 'Pu' or 'Sd' or 'Su'
	ri = density ratio of the two layers (bottom to top)
	si = shear velocity ratio of the two layers (bottom to top)
	vi = compressional-to-shear velocity ratio in top layer
	wi = compressional-to-shear velocity ratio in bottom layer";

plotPSVfluxp::usage = "plotPSVfluxp[acode_String, ri_Real, si_Real, vi_Real, wi_Real]
where
	acode = 'Pd' or 'Pu' or 'Sd' or 'Su'
	ri = density ratio of the two layers (bottom to top)
	si = shear velocity ratio of the two layers (bottom to top)
	vi = compressional-to-shear velocity ratio in top layer
	wi = compressional-to-shear velocity ratio in bottom layer";

plotPSVenergy::usage = "plotPSVenergy[acode_String, ri_Real, vi_Real, wi_Real, smin_Real, smax_Real]
where
	acode = 'Pd' or 'Pu' or 'Sd' or 'Su'
	ri = density ratio of the two layers (bottom to top)
	vi = compressional-to-shear velocity ratio in top layer
	wi = compressional-to-shear velocity ratio in bottom layer
	smin = minimum value of the ratio s to use on the horizontal axis
	smax = maximum value of the ratio s to use on the horizontal axis";

plotPSVenergyp::usage = "plotPSVenergyp[acode_String, ri_Real, vi_Real, wi_Real, smin_Real, smax_Real]
where
	acode = 'Pd' or 'Pu' or 'Sd' or 'Su'
	ri = density ratio of the two layers (bottom to top)
	vi = compressional-to-shear velocity ratio in top layer
	wi = compressional-to-shear velocity ratio in bottom layer
	smin = minimum value of the ratio s to use on the horizontal axis
	smax = maximum value of the ratio s to use on the horizontal axis";

plotPSVfs::usage = "plotPSVfs[acode_String, ui_Real]
where
	acode = 'Pd' or 'Pu' or 'Sd' or 'Su'
	ui = compressional-to-shear ratio of the bottom elastic layer";

coefSH::usage = "coefSH[]";

coefPSV::usage = "coefPSV[]";

coefPSVfs::usage = "coefPSVfs[]";

fluxSH::usage = "fluxSH[]";

fluxPSV::usage = "fluxPSV[]";


Needs["PlotLegends`"];


Begin["`Private`"]

(*Set a line width between thick and thin.*)
regular = AbsoluteThickness[1];


(*This cell plots the Brewster and critical angles for SH waves incident on the boundary of two half-spaces.*)

plotBrewster[smin_Real, smax_Real] := Block[{jb, jc, s, r, ratio, expression, expression1, expression2, expression3, plot1, plot2, plot3, plot4},
         (*Let s be the velocity ratio (bottom/top) and r be the density ratio (bottom/top).  Plot 3 cases for the Brewster angle, assuming 3 different r.  For general r, there are two branches to the line and so two solutions.  But r=1 exactly gives a single line and so needs to be handled differently.*)
         ratio = {0.9, 1.0, 1.1};
         
         expression = Sin[\[Degree] jb] - Sqrt[(s^2 r^2 - 1)/(s^4 r^2 - 1)] /. r -> ratio[[1]];
         solved = Solve[expression == 0, s];
         expression1 = s /. solved;
         plot1 = Plot[{expression1[[2]], expression1[[4]]}, {jb, 0, 90}, PlotRange -> {{0, 90}, {smin, smax}}, PlotStyle -> {{regular, Red, Dashed}, { regular, Red, Dashed}}];
         
         expression = Sin[\[Degree] jb] - Sqrt[(s^2 r^2 - 1)/(s^4 r^2 - 1)] /. r -> ratio[[2]];
         solved = Solve[expression == 0, s];
         expression2 = s /. solved;
         plot2 = Plot[{expression2[[2]]}, {jb, 0, 90}, PlotRange -> {{0, 90}, {smin, smax}}, PlotStyle -> {regular, Green, Dashed}];
         
         expression = Sin[\[Degree] jb] - Sqrt[(s^2 r^2 - 1)/(s^4 r^2 - 1)] /. r -> ratio[[3]];
         solved = Solve[expression == 0, s];
         expression3 = s /. solved;
         plot3 = Plot[{expression3[[2]], expression3[[4]]}, {jb, 0, 90}, PlotRange -> {{0, 90}, {smin, smax}}, PlotStyle -> {{regular, Blue, Dashed}, { regular, Blue, Dashed}}];
         
         (*Plot the critical angle.*)
         plot4 = Plot[1/Sin[\[Degree] jc], {jc, 0, 90}, PlotRange -> {{0, 90}, {smin, smax}}, PlotStyle -> {regular, Black}];
         
         Show[plot1, plot2, plot3, plot4, Frame -> True, FrameStyle -> regular, PlotRangePadding -> None, FrameLabel -> {"S incidence angle in top layer (degrees)", "S velocity ratio (bottom/top)"}, FrameTicks -> {{Table[i 0.5, {i, 0, 5}], None}, {Table[i*15, {i, 0, 6}], None}},
            GridLines -> {Table[i 15, {i, 1, 5}], Table[i 0.5, {i, 1, 4}]}, BaseStyle -> {Medium}, FrameStyle -> regular, TicksStyle -> regular, PlotLabel -> Style["Brewster's angle (dashed)" <> "\n" <> "and critical angle (solid)" <> "\n" <> "density ratio (bottom/top) = " <> ToString[ratio], FontSize -> Medium], AspectRatio -> 1, ImageSize -> 400 ]
         
         ];


(*This cell plots SH reflection and transmission coefficients versus angle of incidence, given the density and velocity ratio of the two layers and a specification of the direction of the incident wave.*)
plotSHcoef[acode_String, ri_Real, si_Real] := Block[{coef, crit, bcode, parms, k, p1, p2, q1, q2, plot1, plot2, \[Gamma]},
         
         (*Gather up the reflection/transmission coefficients into a list of lists.*)
         coef = coefSH[] /. {r -> ri, s -> si};
         
         (*Here is a list of lists of critical angle determinations, to match the order in "coef".*)
         crit = {{1, s}, {1/s, 1}} /. s -> si;
         
         (*Set incident wave.*)
         bcode = {acode <> "Su", acode <> "Sd"};
         
         (*Set the matrix row according to incident wave.*)
         Which[
            	acode == "Sd", k = 1,
            	acode == "Su", k = 2
            ];
         
         (*Define the layer parameters in terms of ratios used in expressions.*)
         parms = Row[{Style["r",Italic]," = " , ri , ", ", Style["s",Italic], " = " , si}];
         
         (*Cut off plotting beyond the critical angle for each scattered wave.*)
         p1 = If[crit[[k, 1]] > 1, If[\[Gamma] \[Degree] > ArcSin[1/crit[[k, 1]]], Indeterminate, Abs[coef[[k, 1]]]], Abs[coef[[k, 1]]]];
         p2 = If[crit[[k, 2]] > 1, If[\[Gamma] \[Degree] > ArcSin[1/crit[[k, 2]]], Indeterminate, Abs[coef[[k, 2]]]], Abs[coef[[k, 2]]]];
         q1 = If[crit[[k, 1]] > 1, If[\[Gamma] \[Degree] > ArcSin[1/crit[[k, 1]]], Indeterminate, Arg[coef[[k, 1]]]], Arg[coef[[k, 1]]]];
         q2 = If[crit[[k, 2]] > 1, If[\[Gamma] \[Degree] > ArcSin[1/crit[[k, 2]]], Indeterminate, Arg[coef[[k, 2]]]], Arg[coef[[k, 2]]]];
         
         plot1 = Plot[{p1, p2}, {\[Gamma], 0, 90}, PlotRange -> {{0, 90}, {0, 2}}, PlotStyle -> {regular,  regular}, BaseStyle -> {Medium}, PlotLabel -> Style[Row[{"scattered wave amplitudes for incident " , acode , "\n" , parms}], Medium], Frame -> True, FrameTicks -> {Table[k 10, {k, 0, 9}], Table[k 0.5, {k, 0, 4}], None, None}, FrameStyle -> regular, FrameLabel -> {"angle of incident wave", "amplitude", "", ""}, PlotLegend -> {Style[bcode[[1]], Medium], Style[bcode[[2]], Medium]}, LegendPosition -> {0.9, -0.25}, LegendSize -> {0.5, 0.5}, LegendShadow -> None, LegendBorder -> regular, ImageSize -> {600, 300}];
         plot2 = Plot[{q1, q2}, {\[Gamma], 0, 90}, PlotRange -> {{0, 90}, {-\[Pi],\[Pi]}}, PlotStyle -> {regular,  regular}, BaseStyle -> {Medium}, PlotLabel -> Style[Row[{"scattered wave phases for incident " , acode , "\n" , parms}], Medium], Frame -> True, FrameTicks -> {Table[k 10, {k, 0, 9}], Table[k \[Pi]/2, {k, -2, 2}], None, None}, FrameStyle -> regular, FrameLabel -> {"angle of incident wave", "phase", "", ""},  PlotLegend -> {Style[bcode[[1]], Medium], Style[bcode[[2]], Medium]}, LegendPosition -> {0.9, -0.25}, LegendSize -> {0.5, 0.5}, LegendShadow -> None, LegendBorder -> regular, ImageSize -> {600, 300}];

		 GraphicsColumn[{plot1,plot2}, Spacings -> Scaled[0.1], ImageSize -> 500] 
         
         ];


(*This cell plots SH energy fluxes versus angle of incidence, given the density and velocity ratio of the two layers and a specification of the direction of the incident wave.*)

plotSHflux[acode_String, ri_Real, si_Real] := Block[{flux, crit, bcode, k, parms, p1, p2, \[Gamma]},
         
         (*Gather up the reflection/transmission coefficients into a list of lists.*)
         flux = fluxSH[] /. {r -> ri, s -> si};
         
         (*Here is a list of lists of critical angle determinations, to match the order in "coef".*)
         crit = {{1, s}, {1/s, 1}} /. s -> si;
         
         (*Set incident wave.*)
         bcode = {acode <> "Su", acode <> "Sd"};
         
         (*Set the matrix row according to incident wave.*)
         Which[
            acode == "Sd", k = 1,
            acode == "Su", k = 2
            ];
         
         (*Define the layer parameters in terms of ratios used in expressions.*)
         parms = Row[{Style["r",Italic]," = " , ri , ", ", Style["s",Italic], " = " , si}];
         
         (*Cut off plotting beyond the critical angle for each scattered wave.*)
         p1 = If[crit[[k, 1]] > 1, If[\[Gamma] \[Degree] > ArcSin[1/crit[[k, 1]]], 0.0, flux[[k, 1]]], flux[[k, 1]]];
         p2 = If[crit[[k, 2]] > 1, If[\[Gamma] \[Degree] > ArcSin[1/crit[[k, 2]]], 0.0, flux[[k, 2]]], flux[[k, 2]]];
         
         Plot[{p1, p2}, {\[Gamma], 0, 90}, PlotRange -> {{0, 90}, {0, 1}}, PlotStyle -> {regular, regular}, BaseStyle -> {Medium}, PlotLabel -> Style[Row[{"scattered energy fluxes for incident " , acode , "\n" , parms}], Medium], Frame -> True, FrameTicks -> {Table[k 10, {k, 0, 9}], Table[k 0.25, {k, 0, 4}], None, None}, FrameStyle -> regular, FrameLabel -> {"angle of incident wave", "amplitude", "", ""}, PlotLegend -> {Style[bcode[[1]], Medium], Style[bcode[[2]], Medium]}, LegendPosition -> {0.9, -0.25}, LegendSize -> {0.5, 0.5}, LegendShadow -> None, LegendBorder -> regular, ImageSize -> {480, 240}]
         
         ];


(*This cell plots the SH energy flux in the two scattered waves as contour plots over incident angle and velocity ratio..*)

plotSHenergy[acode_String, ri_Real, smin_Real, smax_Real] := Block[{flux, crit, bcode, k, parms, colors, plot, plot1, plot2, p1, p2, \[Gamma], legend},
         
         (*Gather up the reflection/transmission coefficients into a list of lists.*)
         flux = fluxSH[] /. r -> ri;
         
         (*Here is a list of lists of critical angle determinations, to match the order in "coef".*)
         crit = {{1, s}, {1/s, 1}};
         
         bcode = {acode <> "Su", acode <> "Sd"};
         
         (*Set the matrix row according to incident wave.*)
         Which[
            acode == "Sd", k = 1,
            acode == "Su", k = 2
            ];
         
         parms = Row[{Style["r",Italic]," = " , ri }];
         
         colors = "TemperatureMap";
         
         (*Cut off plotting beyond the critical angle for each scattered wave.*)
         p1 = If[crit[[k, 1]] > 1, If[\[Gamma] \[Degree] > ArcSin[1/crit[[k, 1]]], 0.0, flux[[k, 1]]], flux[[k, 1]]];
         p2 = If[crit[[k, 2]] > 1, If[\[Gamma] \[Degree] > ArcSin[1/crit[[k, 2]]], 0.0, flux[[k, 2]]], flux[[k, 2]]];
         
         (*Plot the energy in the top layer.*)
         plot1 = ContourPlot[p1, {\[Gamma], 0, 90}, {s, smin, smax}, Contours -> Table[0.1 i, {i, 1, 9}], ContourStyle -> Black, ColorFunction -> colors, ColorFunctionScaling -> False, PlotRangePadding -> None, FrameLabel -> {"angle of incident wave", Row[{Style["S",Italic]," velocity ratio (bottom/top)"}]}, FrameTicks -> {{Table[i 0.5, {i, 0, Round[smax/0.5]}], None}, {Table[i 15, {i, 0, 6}], None}}, PlotLabel ->Row[{ bcode[[1]] , " energy flux" , "\n" , parms}], BaseStyle -> {Medium}, FrameStyle -> regular, TicksStyle -> regular, Mesh -> {5, 4}, MeshFunctions -> {#1 &, #2 &}, MeshStyle -> {{regular, Gray}, {regular, Gray}},ImageSize->210];
         
         (*Plot the energy in the bottom layer.*)
         plot2 = ContourPlot[p2, {\[Gamma], 0, 90}, {s, smin, smax}, Contours -> Table[0.1 i, {i, 1, 9}], ContourStyle -> Black, ColorFunction -> colors, ColorFunctionScaling -> False, PlotRangePadding -> None, FrameLabel -> {"angle of incident wave", Row[{Style["S",Italic]," velocity ratio (bottom/top)"}]}, FrameTicks -> {{Table[i 0.5, {i, 0, Round[smax/0.5]}], None}, {Table[i 15, {i, 0, 6}], None}}, PlotLabel -> Row[{ bcode[[1]] , " energy flux" , "\n" , parms}], BaseStyle -> {Medium}, FrameStyle -> regular, TicksStyle -> regular, Mesh -> {5, 4}, MeshFunctions -> {#1 &, #2 &}, MeshStyle -> {{regular, Gray}, {regular, Gray}},ImageSize->210];

         legend = ContourPlot[y,{x,0,1},{y,0,1},ColorFunction -> colors,AspectRatio->10,Ticks->{None,Automatic},Frame->None,ImageSize->{20,150},Epilog->{Style[Text[0,{.5,.05}],10],Style[Text[1,{.5,.95}],10]}];

         
         Row[{plot1, plot2, legend},Graphics[{},ImageSize->{20,150}]]
         
         ];


(*This cell plots P-SV reflection and transmission coefficients versus angle of incidence, given the density and velocity ratio between the two layers, the compressional-to-shear velocity ratios of the layers, and a specification of the direction of the incident wave.*)

plotPSVcoef[acode_String, ri_Real, si_Real, vi_Real, wi_Real] := Block[{coef, crit, bcode, k, parms, p1, p2, p3, p4, q1, q2, q3, q4, plot1, plot2, \[Gamma]},
            
            (*Gather up the reflection/transmission coefficients into a list of lists.*)
            coef = coefPSV[] /. {r -> ri, s -> si, v -> vi, w -> wi};
            
            (*Here is a list of lists of critical angle determinations, to match the order in "coef".*)
            crit = {{1, 1/v, (w s)/v, s/v}, {v, 1, w s, s}, {v/(w s), 1/(w s), 1, 1/w}, {v/s, 1/s, w, 1}} /. {s -> si, v -> vi, w -> wi};
            
            (*Set incident wave.*)
            bcode = {acode <> "Pu", acode <> "Su", acode <> "Pd", acode <> "Sd"};
            
            (*Set the matrix row according to incident wave.*)
            Which[
                acode == "Pd", k = 1,
                acode == "Sd", k = 2,
                acode == "Pu", k = 3,
                acode == "Su", k = 4
                ];
            
            (*Define the layer parameters in terms of ratios used in expressions.*)
            parms = Row[{Style["r",Italic]," = " , ri , ", ", Style["s",Italic], " = " , si, "\n", Style["v",Italic], " = " , vi,", ", Style["w",Italic], " = " , wi}];
            
            (*Cut off plotting beyond the critical angle for each scattered wave.*)
            p1 = If[crit[[k, 1]] > 1, If[\[Gamma] \[Degree] > ArcSin[1/crit[[k, 1]]], Indeterminate, Abs[coef[[k, 1]]]], Abs[coef[[k, 1]]]];
            p2 = If[crit[[k, 2]] > 1, If[\[Gamma] \[Degree] > ArcSin[1/crit[[k, 2]]], Indeterminate, Abs[coef[[k, 2]]]], Abs[coef[[k, 2]]]];
            p3 = If[crit[[k, 3]] > 1, If[\[Gamma] \[Degree] > ArcSin[1/crit[[k, 3]]], Indeterminate, Abs[coef[[k, 3]]]], Abs[coef[[k, 3]]]];
            p4 = If[crit[[k, 4]] > 1, If[\[Gamma] \[Degree] > ArcSin[1/crit[[k, 4]]], Indeterminate, Abs[coef[[k, 4]]]], Abs[coef[[k, 4]]]];
            q1 = If[crit[[k, 1]] > 1, If[\[Gamma] \[Degree] > ArcSin[1/crit[[k, 1]]], Indeterminate, Arg[coef[[k, 1]]]], Arg[coef[[k, 1]]]];
            q2 = If[crit[[k, 2]] > 1, If[\[Gamma] \[Degree] > ArcSin[1/crit[[k, 2]]], Indeterminate, Arg[coef[[k, 2]]]], Arg[coef[[k, 2]]]];
            q3 = If[crit[[k, 3]] > 1, If[\[Gamma] \[Degree] > ArcSin[1/crit[[k, 3]]], Indeterminate, Arg[coef[[k, 3]]]], Arg[coef[[k, 3]]]];
            q4 = If[crit[[k, 4]] > 1, If[\[Gamma] \[Degree] > ArcSin[1/crit[[k, 4]]], Indeterminate, Arg[coef[[k, 4]]]], Arg[coef[[k, 4]]]];
             
            plot1 = Plot[{p1, p2, p3, p4}, {\[Gamma], 0, 90}, PlotRange -> {{0, 90}, {0, 2}}, PlotStyle -> {regular, regular, regular, regular}, BaseStyle -> {Medium}, PlotLabel -> Style[Row[{"scattered wave amplitudes for incident " , acode , "\n" , parms}], Medium], Frame -> True, FrameTicks -> {Table[k 10, {k, 0, 9}], Table[k 0.5, {k, 0, 4}], None, None}, FrameStyle -> regular, FrameLabel -> {"angle of incident wave", "amplitude", "", ""}, PlotLegend -> {Style[bcode[[1]], Medium], Style[bcode[[2]], Medium], Style[bcode[[3]], Medium], Style[bcode[[4]], Medium]}, LegendPosition -> {0.9, -0.25}, LegendSize -> {0.5, 0.5}, LegendShadow -> None, LegendBorder -> regular, ImageSize -> {480, 240}];
            plot2 = Plot[{q1, q2, q3, q4}, {\[Gamma], 0, 90}, PlotRange -> {{0, 90}, {-\[Pi], \[Pi]}}, PlotStyle -> {regular, regular, regular, regular}, BaseStyle -> {Medium}, PlotLabel -> Style[Row[{"scattered wave phases for incident " , acode , "\n" , parms}], Medium], Frame -> True, FrameTicks -> {Table[k 10, {k, 0, 9}], Table[k \[Pi]/2, {k, -2, 2}], None, None}, FrameStyle -> regular, FrameLabel -> {"angle of incident wave", "phase", "", ""},  PlotLegend -> {Style[bcode[[1]], Medium], Style[bcode[[2]], Medium], Style[bcode[[3]], Medium], Style[bcode[[4]], Medium]}, LegendPosition -> {0.9, -0.25}, LegendSize -> {0.5, 0.5}, LegendShadow -> None, LegendBorder -> regular, ImageSize -> {480, 240}];

            GraphicsColumn[{plot1, plot2}, Spacings -> Scaled[0.1]]

            ];


(*This cell plots P-SV energy fluxes versus angle of incidence, given the density and velocity ratio between the two layers, the compressional-to-shear velocities of the two layers, and a specification of the direction of the incident wave.*)

plotPSVflux[acode_String, ri_Real, si_Real, vi_Real, wi_Real] := Block[{flux, crit, bcode, k, parms, p1, p2, p3, p4, \[Gamma]},
            
            (*Gather up the reflection/transmission coefficients into a list of lists.*)
            flux = fluxPSV[] /. {r -> ri, s -> si, v -> vi, w -> wi};
            
            (*Here is a list of lists of critical angle determinations, to match the order in "coef".*)
            crit = {{1, 1/v, (w s)/v, s/v}, {v, 1, w s, s}, {v/(w s), 1/(w s), 1, 1/w}, {v/s, 1/s, w, 1}} /. {s -> si, v -> vi, w -> wi};
            
            (*Set incident wave.*)
            bcode = {acode <> "Pu", acode <> "Su", acode <> "Pd", acode <> "Sd"};
            
            (*Set the matrix row according to incident wave.*)
            Which[
                acode == "Pd", k = 1,
                acode == "Sd", k = 2,
                acode == "Pu", k = 3,
                acode == "Su", k = 4
                ];
            
            parms = Row[{Style["r",Italic]," = " , ri , ", ", Style["s",Italic], " = " , si, "\n", Style["v",Italic], " = " , vi,", ", Style["w",Italic], " = " , wi}];
            
            (*Cut off plotting beyond the critical angle for each scattered wave.*)
            p1 = If[crit[[k, 1]] > 1, If[\[Gamma] \[Degree] > ArcSin[1/crit[[k, 1]]], 0.0, flux[[k, 1]]], flux[[k, 1]]];
            p2 = If[crit[[k, 2]] > 1, If[\[Gamma] \[Degree] > ArcSin[1/crit[[k, 2]]], 0.0, flux[[k, 2]]], flux[[k, 2]]];
            p3 = If[crit[[k, 3]] > 1, If[\[Gamma] \[Degree] > ArcSin[1/crit[[k, 3]]], 0.0, flux[[k, 3]]], flux[[k, 3]]];
            p4 = If[crit[[k, 4]] > 1, If[\[Gamma] \[Degree] > ArcSin[1/crit[[k, 4]]], 0.0, flux[[k, 4]]], flux[[k, 4]]];
            
            (*Plot individual fluxes.*)
            Plot[{p1, p2, p3, p4}, {\[Gamma], 0, 90}, PlotRange -> {{0, 90}, {0, 1}}, PlotStyle -> {regular, regular, regular, regular}, BaseStyle -> {Medium}, PlotLabel -> Style[Row[{"scattered energy fluxes for incident " , acode , "\n" , parms}], Medium], Frame -> True, FrameTicks -> {Table[k 10, {k, 0, 9}], Table[k 0.25, {k, 0, 4}], None, None}, FrameStyle -> regular, FrameLabel -> {"angle of incident wave", "energy flux", "", ""}, PlotLegend -> {Style[bcode[[1]], Medium], Style[bcode[[2]], Medium], Style[bcode[[3]], Medium], Style[bcode[[4]], Medium]}, LegendPosition -> {0.9, -0.25}, LegendSize -> {0.5, 0.5}, LegendShadow -> None, LegendBorder -> regular, ImageSize -> {500, 250}]
            
            ];


(*This cell plots P-SV energy fluxes versus angle of incidence, given the density and velocity ratio between the two layers, the compressional-to-shear velocities of the two layers, and a specification of the direction of the incident wave.  The fluxes are partitioned into top and bottom contributions and into P and SV contributions.*)

plotPSVfluxp[acode_String, ri_Real, si_Real, vi_Real, wi_Real] := Block[{flux, crit, bcode, ccode, k, parms, p1, p2, p3, p4, q1, q2, q3, q4, \[Gamma]},
            
            (*Gather up the reflection/transmission coefficients into a list of lists.*)
            flux = fluxPSV[] /. {r -> ri, s -> si, v -> vi, w -> wi};
            
            (*Here is a list of lists of critical angle determinations, to match the order in "coef".*)
            crit = {{1, 1/v, (w s)/v, s/v}, {v, 1, w s, s}, {v/(w s), 1/(w s), 1, 1/w}, {v/s, 1/s, w, 1}} /. {s -> si, v -> vi, w -> wi};
            
            (*Set incident wave.*)
            bcode = {acode <> "Pu", acode <> "Su", acode <> "Pd", acode <> "Sd"};
            
            (*Set the matrix row according to incident wave.*)
            Which[
                acode == "Pd", k = 1,
                acode == "Sd", k = 2,
                acode == "Pu", k = 3,
                acode == "Su", k = 4
                ];
            
            parms = Row[{Style["r",Italic]," = " , ri , ", ", Style["s",Italic], " = " , si, "\n", Style["v",Italic], " = " , vi,", ", Style["w",Italic], " = " , wi}];
            
            (*Cut off plotting beyond the critical angle for each scattered wave.*)
            p1 = If[crit[[k, 1]] > 1, If[\[Gamma] \[Degree] > ArcSin[1/crit[[k, 1]]], 0.0, flux[[k, 1]]], flux[[k, 1]]];
            p2 = If[crit[[k, 2]] > 1, If[\[Gamma] \[Degree] > ArcSin[1/crit[[k, 2]]], 0.0, flux[[k, 2]]], flux[[k, 2]]];
            p3 = If[crit[[k, 3]] > 1, If[\[Gamma] \[Degree] > ArcSin[1/crit[[k, 3]]], 0.0, flux[[k, 3]]], flux[[k, 3]]];
            p4 = If[crit[[k, 4]] > 1, If[\[Gamma] \[Degree] > ArcSin[1/crit[[k, 4]]], 0.0, flux[[k, 4]]], flux[[k, 4]]];
            
            (*Plot partitioned fluxes -- top and bottom, P and SV.*)
            q1 = p1 + p2;
            q2 = p3 + p4;
            q3 = p1 + p3;
            q4 = p2 + p4;
            ccode = {"top", "bottom", "P", "S"};
             
            Plot[{q1, q2, q3, q4}, {\[Gamma], 0, 90}, PlotRange -> {{0, 90}, {0, 1}}, PlotStyle -> {regular, regular, regular, regular}, BaseStyle -> {Medium}, PlotLabel -> Style[Row[{"scattered energy fluxes for incident " , acode , "\n" , parms}],   Medium], Frame -> True, FrameTicks -> {Table[k 10, {k, 0, 9}], Table[k 0.25, {k, 0, 4}], None, None}, FrameStyle ->  regular, FrameLabel -> {"angle of incident wave", "energy flux", "", ""}, PlotLegend -> {Style[ccode[[1]], Medium], Style[ccode[[2]], Medium], Style[ccode[[3]], Medium], Style[ccode[[4]], Medium]}, LegendPosition -> {0.9, -0.25}, LegendSize -> {0.5, 0.5}, LegendShadow -> None, LegendBorder -> regular, ImageSize -> {500, 250}]
         
            ];


(*This cell plots the PSV energy flux in the four scattered waves as contour plots over incident angle and velocity ratio.*)

plotPSVenergy[acode_String, ri_Real, vi_Real, wi_Real, smin_Real, smax_Real] := Block[{flux, crit, bcode, colors, k, parms, plot, aplot, line, linePlot1, linePlot2, linePlot3, linePlot4, p1, p2, p3, p4, q, \[Gamma], legend},
      
      (*Gather up the reflection/transmission coefficients into a list of lists.*)
      flux = fluxPSV[] /. {r -> ri, v -> vi, w -> wi};
      
      (*Here is a list of lists of critical angle determinations, to match the order in "coef".*)
      crit = {{1, 1/v, (w s)/v, s/v}, {v, 1, w s, s}, {v/(w s), 1/(w s), 1, 1/w}, {v/s, 1/s, w, 1}} /. {v -> vi, w -> wi};
      
      (*Set incident wave.*)
      bcode = {acode <> "Pu", acode <> "Su", acode <> "Pd", acode <> "Sd"};
      
      (*Set the matrix row according to incident wave.*)
      Which[
        acode == "Pd", k = 1,
        acode == "Sd", k = 2,
        acode == "Pu", k = 3,
        acode == "Su", k = 4
        ];
      
      parms = Row[{Style["r",Italic]," = " , ri ,  "\n", Style["v",Italic], " = " , vi,", ", Style["w",Italic], " = " , wi}];
      
      colors = "TemperatureMap";
      
      (*Plot the line relating to ArcSin (b1/a1).  For a Poisson solid, this angle is ArcSin(1/Sqrt[3]) = 35.26 degrees.  For non-Poisson solids, the value is greater or less, depending on the value of sigma.*)
      line = {{ArcSin[1/vi]/\[Degree], 0}, {ArcSin[1/vi]/Degree, smax}};
      linePlot1 = ListPlot[line, Joined -> True, PlotStyle -> {regular, Gray, Dashed}];
      (*Plot the line relating to ArcSin (b1/a2).*)
      linePlot2 = Plot[1/(wi Sin[\[Gamma] \[Degree]]), {\[Gamma], 0, 90}, PlotRange -> {{0, 90}, {smin, smax}}, PlotStyle -> {regular, Gray, Dashed}];
      (*Plot the line relating to ArcSin(b1/b2), or equivalently ArcSin(a1/a2) if Poisson ratios are same.*)
      linePlot3 = Plot[1/(Sin[\[Gamma] \[Degree]]), {\[Gamma], 0, 90}, PlotRange -> {{0, 90}, {smin, smax}}, PlotStyle -> {regular, Gray, Dashed}];
      (*Plot the line relating to ArcSin (a1/b2).*)
      linePlot4 = Plot[vi/Sin[\[Gamma] \[Degree]], {\[Gamma], 0, 90}, PlotRange -> {{0, 90}, {smin, smax}}, PlotStyle -> {regular, Gray, Dashed}];
      
      
      (*Cut off plotting beyond the critical angle for each scattered wave.*)
      p1 = If[crit[[k, 1]] > 1, If[\[Gamma] \[Degree] > ArcSin[1/crit[[k, 1]]], 0.0, Abs[flux[[k, 1]]]], Abs[flux[[k, 1]]]];
      p2 = If[crit[[k, 2]] > 1, If[\[Gamma] \[Degree] > ArcSin[1/crit[[k, 2]]], 0.0, Abs[flux[[k, 2]]]], Abs[flux[[k, 2]]]];
      p3 = If[crit[[k, 3]] > 1, If[\[Gamma] \[Degree] > ArcSin[1/crit[[k, 3]]], 0.0, Abs[flux[[k, 3]]]], Abs[flux[[k, 3]]]];
      p4 = If[crit[[k, 4]] > 1, If[\[Gamma] \[Degree] > ArcSin[1/crit[[k, 4]]], 0.0, Abs[flux[[k, 4]]]], Abs[flux[[k, 4]]]];
      
      (*Plot the energy in each scattered wave.*)
      
      plot = Table[0, {j, 1, 4}];
            
      Do[
        Which[
          j == 1, q = p1,
          j == 2, q = p2,
          j == 3, q = p3,
          j == 4, q = p4
          ];
        aplot = ContourPlot[q, {\[Gamma], 0, 90}, {s, smin, smax}, Contours -> Table[0.1 i, {i, 1, 9}], ContourStyle -> Black, ColorFunction -> colors, ColorFunctionScaling -> False, PlotRangePadding -> None, FrameLabel -> {"angle of incident wave", Row[{Style["S",Italic]," velocity ratio (bottom/top)"}]}, FrameTicks -> {{Table[i 0.5, {i, 0, Round[smax/0.5]}], None}, {Table[i 15, {i, 0, 6}], None}}, PlotLabel -> Row[{bcode[[j]] , " energy flux" , "\n" , parms}], BaseStyle -> {Medium}, FrameStyle -> regular, TicksStyle -> regular, Mesh -> {5, 4}, MeshFunctions -> {#1 &, #2 &}, MeshStyle -> {{regular, Gray}, {regular, Gray}},ImageSize->210];
        plot[[j]] = Show[aplot, linePlot1, linePlot2, linePlot3, linePlot4],
        {j, 1, 4}
        ];
      legend = ContourPlot[y,{x,0,1},{y,0,1},ColorFunction -> colors,AspectRatio->10,Ticks->{None,Automatic},Frame->None,ImageSize->{20,150},Epilog->{Style[Text[0,{.5,.05}],10],Style[Text[1,{.5,.95}],10]}];
      Row[{
         Grid[{
           {plot[[1]],Graphics[{},ImageSize->{20,150}], plot[[2]]}, 
           {plot[[3]],Graphics[{},ImageSize->{20,150}], plot[[4]]}}
         ],
        Graphics[{},ImageSize->{20,150}], 
        legend
     }]
      
      ];


(*This cell plots the partition of PSV energy flux in top and bottom layers and in P and SV waves as contour plots over incident angle and velocity ratio.*)

plotPSVenergyp[acode_String, ri_Real, vi_Real, wi_Real, smin_Real, smax_Real] := Block[{flux, crit, bcode, ccode, colors, k, parms, plot, aplot, line, linePlot1, linePlot2, linePlot3, linePlot4, p1, p2, p3, p4, q, \[Gamma], legend},
      
      (*Gather up the reflection/transmission coefficients into a list of lists.*)
      flux = fluxPSV[] /. {r -> ri, v -> vi, w -> wi};
      
      (*Here is a list of lists of critical angle determinations, to match the order in "coef".*)
      crit = {{1, 1/v, (w s)/v, s/v}, {v, 1, w s, s}, {v/(w s), 1/(w s), 1, 1/w}, {v/s, 1/s, w, 1}} /. {v -> vi, w -> wi};
      
      (*Set incident wave.*)
      bcode = {acode <> "Pu", acode <> "Su", acode <> "Pd", acode <> "Sd"};
      
      (*Set the matrix row according to incident wave.*)
      Which[
        acode == "Pd", k = 1,
        acode == "Sd", k = 2,
        acode == "Pu", k = 3,
        acode == "Su", k = 4
        ];
      
      parms = Row[{Style["r",Italic]," = " , ri ,  "\n", Style["v",Italic], " = " , vi,", ", Style["w",Italic], " = " , wi}];
      
      colors = "TemperatureMap";
      
      (*Plot the line relating to ArcSin (b1/a1).  For a Poisson solid, this angle is ArcSin(1/Sqrt[3]) = 35.26 degrees.  For non-Poisson solids, the value is greater or less, depending on the value of sigma.*)
      line = {{ArcSin[1/vi]/\[Degree], 0}, {ArcSin[1/vi]/Degree, smax}};
      linePlot1 = ListPlot[line, Joined -> True, PlotStyle -> {regular, Gray, Dashed}];
      (*Plot the line relating to ArcSin (b1/a2).*)
      linePlot2 = Plot[1/(wi Sin[\[Gamma] \[Degree]]), {\[Gamma], 0, 90}, PlotRange -> {{0, 90}, {smin, smax}}, PlotStyle -> {regular, Gray, Dashed}];
      (*Plot the line relating to ArcSin(b1/b2), or equivalently ArcSin(a1/a2) if Poisson ratios are same.*)
      linePlot3 = Plot[1/(Sin[\[Gamma] \[Degree]]), {\[Gamma], 0, 90}, PlotRange -> {{0, 90}, {smin, smax}}, PlotStyle -> {regular, Gray, Dashed}];
      (*Plot the line relating to ArcSin (a1/b2).*)
      linePlot4 = Plot[vi/Sin[\[Gamma] \[Degree]], {\[Gamma], 0, 90}, PlotRange -> {{0, 90}, {smin, smax}}, PlotStyle -> {regular, Gray, Dashed}];
      
      
      (*Cut off plotting beyond the critical angle for each scattered wave.*)
      p1 = If[crit[[k, 1]] > 1, If[\[Gamma] \[Degree] > ArcSin[1/crit[[k, 1]]], 0.0, Abs[flux[[k, 1]]]], Abs[flux[[k, 1]]]];
      p2 = If[crit[[k, 2]] > 1, If[\[Gamma] \[Degree] > ArcSin[1/crit[[k, 2]]], 0.0, Abs[flux[[k, 2]]]], Abs[flux[[k, 2]]]];
      p3 = If[crit[[k, 3]] > 1, If[\[Gamma] \[Degree] > ArcSin[1/crit[[k, 3]]], 0.0, Abs[flux[[k, 3]]]], Abs[flux[[k, 3]]]];
      p4 = If[crit[[k, 4]] > 1, If[\[Gamma] \[Degree] > ArcSin[1/crit[[k, 4]]], 0.0, Abs[flux[[k, 4]]]], Abs[flux[[k, 4]]]];
      
      (*Plot the partitions of energy for top and bottom and for P and S.*)
      ccode = {"top", "bottom", "P", "S"};
      plot = Table[0, {j, 1, 4}]; 
       
      Do[
        Which[
          j == 1, q = p1 + p2,
          j == 2, q = p3 + p4,
          j == 3, q = p1 + p3,
          j == 4, q = p2 + p4
          ];
        aplot = ContourPlot[q, {\[Gamma], 0, 90}, {s, 0.0, smax}, Contours -> Table[0.1 i, {i, 1, 9}], ContourStyle -> Black, ColorFunction -> colors, ColorFunctionScaling -> False, PlotRangePadding -> None, FrameLabel -> {"angle of incident wave", Row[{Style["S",Italic]," velocity ratio (bottom/top)"}]}, FrameTicks -> {{Table[i 0.5, {i, 0, Round[smax/0.5]}], None}, {Table[i 15, {i, 0, 6}], None}}, PlotLabel -> Row[{ccode[[j]] , " energy flux" , "\n" , parms}], BaseStyle -> {Medium}, FrameStyle -> regular, TicksStyle -> regular, Mesh -> {5, 4}, MeshFunctions -> {#1 &, #2 &}, MeshStyle -> {{regular, Gray}, {regular, Gray}}, ImageSize -> 210];
        plot[[j]] = Show[aplot, linePlot1, linePlot2, linePlot3, linePlot4],
        {j, 1, 4}
        ];
      legend = ContourPlot[y,{x,0,1},{y,0,1},ColorFunction -> colors,AspectRatio->10,Ticks->{None,Automatic},Frame->None,ImageSize->{20,150},Epilog->{Style[Text[0,{.5,.05}],10],Style[Text[1,{.5,.95}],10]}];
      Row[{
         Grid[{
           {plot[[1]],Graphics[{},ImageSize->{20,150}], plot[[2]]}, 
           {plot[[3]],Graphics[{},ImageSize->{20,150}], plot[[4]]}}
         ],
        Graphics[{},ImageSize->{20,150}], 
        legend
     }]

     ];



(*This cell plots the P and SV coefficients for elastic waves incident on a boundary with a vacuum.*)

plotPSVfs[acode_String, ui_Real] := Block[{coef, crit, bcode, parms, k, p1, p2, q1, q2, max1, max2, maxa, plot1, plot2, \[Gamma]},
      
      (*Gather up the reflection/transmission coefficients into a list of lists.*)
      coef = coefPSVfs[] /. u -> ui;
      
      (*Here is a list of lists of critical angle determinations, to match the order in "coef".*)
      crit = {{1, 1/u}, {u, 1}} /. u -> ui;

      (*Set incident wave.*)
      bcode = {acode <> "Pd", acode <> "Sd"};
      
      (*Set the matrix row according to incident wave.*)
      Which[
        acode == "Pu", k = 1,
        acode == "Su", k = 2
        ];
      
      (*Define the layer parameters in terms of ratios used in expressions.*)
      parms = Row[{Style["u",Italic]," = " , ui}];
      
      (*Cut off plotting beyond the critical angle for each scattered wave.*)
      p1 = If[crit[[k, 1]] > 1, If[\[Gamma] \[Degree] > ArcSin[1/crit[[k, 1]]], Indeterminate, Abs[coef[[k, 1]]]], Abs[coef[[k, 1]]]];
      p2 = If[crit[[k, 2]] > 1, If[\[Gamma] \[Degree] > ArcSin[1/crit[[k, 2]]], Indeterminate, Abs[coef[[k, 2]]]], Abs[coef[[k, 2]]]];
      q1 = If[crit[[k, 1]] > 1, If[\[Gamma] \[Degree] > ArcSin[1/crit[[k, 1]]], Indeterminate, Arg[coef[[k, 1]]]], Arg[coef[[k, 1]]]];
      q2 = If[crit[[k, 2]] > 1, If[\[Gamma] \[Degree] > ArcSin[1/crit[[k, 2]]], Indeterminate, Arg[coef[[k, 2]]]], Arg[coef[[k, 2]]]];

      (*Plot the amplitudes.  The amplitudes may be > 1, and so the PlotRange for the vertical axis may need to be varied.  GridLines and FrameTicks may also need adjustment.  Just tabulate p1 and p2 at a fine increment and find the maximum.
      max1 = Ceiling[Max[Select[Table[p1, {\[Gamma], 0, 90, 0.01}], NumberQ]]];
      max2 = Ceiling[Max[Select[Table[p2, {\[Gamma], 0, 90, 0.01}], NumberQ]]];
      maxa = Max[max1, max2];*)
      
      plot1 = Plot[{p1, p2}, {\[Gamma], 0, 90}, PlotRange -> {{0, 90}, {0, 2}}, PlotStyle -> {regular, regular}, BaseStyle -> {Medium}, PlotLabel -> Style[Row[{"scattered wave amplitudes for incident " , acode , "\n" , parms}], Medium], Frame -> True, FrameTicks -> {Table[k 10, {k, 0, 9}], Table[k 1.0, {k, 0, 2}], None, None}, FrameStyle -> regular, FrameLabel -> {"angle of incident wave", "amplitude", "", ""}, PlotLegend -> {Style[bcode[[1]], Medium], Style[bcode[[2]], Medium]}, LegendPosition -> {0.9, -0.25}, LegendSize -> {0.5, 0.5}, LegendShadow -> None, LegendBorder -> regular, ImageSize -> {500, 250}];
      plot2 = Plot[{q1, q2}, {\[Gamma], 0, 90}, PlotRange -> {{0, 90}, {-\[Pi], \[Pi]}}, PlotStyle -> {regular, regular}, BaseStyle -> {Medium}, PlotLabel -> Style[Row[{"scattered wave phases for incident " , acode , "\n" , parms}], Medium], Frame -> True, FrameTicks -> {Table[k 10, {k, 0, 9}], Table[k \[Pi]/2, {k, -2, 2}], None, None}, FrameStyle -> regular, FrameLabel -> {"angle of incident wave", "phase", "", ""}, PlotLegend -> {Style[bcode[[1]], Medium], Style[bcode[[2]], Medium]}, LegendPosition -> {0.9, -0.25}, LegendSize -> {0.5, 0.5}, LegendShadow -> None, LegendBorder -> regular, ImageSize -> {500, 250}];

      GraphicsColumn[{plot1,plot2}, Spacings -> Scaled[0.1]]

      ];


(*
P-SV coefficients
This cell expresses the reflection and transmission coefficients of plane P-SV waves in terms of layer parameter ratios: r for density ratio of layer 2 over layer 1,  s for shear velocity ratio of layer 2 over layer 1, v for ratio of compressional/shear velocity in the top layer, and w for similar ratio in the bottom layer.  The expressions should be evaluated for specific incidence angles \[Gamma] (degrees) and for specific v, w, r, s.  The starting point are the expressions in Aki & Richards (eq. 5.40).  The waves are incident on the plane boundary of two homogeneous half-spaces where subscripts 1 and 2 refer to the upper and lower media.  The notebook creates sixteen expressions for calculation of reflection/transmission coefficients of the sixteen scattered waves from the 4 possible incident waves (Pd,Sd,Pu,Su).  For instance, PdPu is the scattered Pu from an incident Pd.  After making this set of definitions, they are returned in a matrix (list of lists) like:
{
{PdPu,PdSu,PdPd,PdSd},
{SdPu,SdSu,SdPd,SdSd},
{PuPu,PuSu,PuPd,PuSd},
{SuPu,SuSu,SuPd,SuSd}
}
*)

coefPSV[] := Block[{r, s, v, w, \[Alpha]1, \[Alpha]2, \[Beta]1, \[Beta]2, \[Rho]1, \[Rho]2, i1, i2, j1, j2, csi1, csi2, csj1, csj2, a, b, c, d, e, f, g, h, dd, p, factor, \[Gamma]},
         
         (*Set up substitutions such that final expressions are in terms of v, w, r, s.*)
         \[Alpha]1 = \[Beta]1 v;(*This eliminates \[Alpha]1 from the equations.*)
         \[Alpha]2 = \[Beta]2 w;(*This eliminates \[Alpha]2 from the equations.*)
         \[Rho]2 = r \[Rho]1;(*This eliminates both densities as \[Rho]1 eventually cancels everywhere.*)
         \[Beta]2 = s \[Beta]1;(*This eliminates both velocities as \[Beta]1 eventually cancels everywhere.*)
         
         (*Pd expressions*)
         csi1 = Sqrt[1 - Sin[i1]^2];
         csj1 = Sqrt[1 - (1/v^2) Sin[i1]^2];
         csi2 = Sqrt[1 - (w^2/v^2) s^2 Sin[i1]^2];
         csj2 = Sqrt[1 - (1/v^2) s^2 Sin[i1]^2];
         a = \[Rho]2 (1 - 2 \[Beta]2^2 p^2) - \[Rho]1 (1 - 2 \[Beta]1^2 p^2);
         b = \[Rho]2 (1 - 2 \[Beta]2^2 p^2) + 2 \[Rho]1 \[Beta]1^2 p^2;
         c = \[Rho]1 (1 - 2 \[Beta]1^2 p^2) + 2 \[Rho]2 \[Beta]2^2 p^2;
         d = 2 (\[Rho]2 \[Beta]2^2 - \[Rho]1 \[Beta]1^2);
         e = b csi1/\[Alpha]1 + c csi2/\[Alpha]2;
         f = b csj1/\[Beta]1 + c csj2/\[Beta]2;
         g = a - d csi1 csj2/(\[Alpha]1 \[Beta]2);
         h = a - d csi2 csj1/(\[Alpha]2 \[Beta]1);
         dd = e f + g h p^2;
         
         PdPu = ((b csi1/\[Alpha]1 - c csi2/\[Alpha]2) f - h p^2 (a + d csi1 csj2/(\[Alpha]1 \[Beta]2)))/dd;
         PdPu = Cancel[PdPu];
         PdPu = PdPu /. p -> Sin[i1]/\[Alpha]1;
         PdPu = Cancel[PdPu];
         PdPu = PdPu /. i1 -> \[Gamma] \[Degree];
         
         PdSu = -2 csi1 (a b + c d csi2 csj2/(\[Alpha]2 \[Beta]2)) p/(\[Beta]1 dd);
         PdSu = Cancel[PdSu];
         PdSu = PdSu /. p -> Sin[i1]/\[Alpha]1;
         PdSu = Cancel[PdSu];
         PdSu = PdSu /. i1 -> \[Gamma] \[Degree];
         
         PdPd = 2 \[Rho]1 csi1 f/(\[Alpha]2 dd);
         PdPd = Cancel[PdPd];
         PdPd = PdPd /. p -> Sin[i1]/\[Alpha]1;
         PdPd = Cancel[PdPd];
         PdPd = PdPd /. i1 -> \[Gamma] \[Degree];
         
         PdSd = 2 \[Rho]1 csi1 h p/(\[Beta]2 dd);
         PdSd = Cancel[PdSd];
         PdSd = PdSd /. p -> Sin[i1]/\[Alpha]1;
         PdSd = Cancel[PdSd];
         PdSd = PdSd /. i1 -> \[Gamma] \[Degree];
         
         (*Sd expressions*)
         csi1 = Sqrt[1 - v^2 Sin[j1]^2];
         csj1 = Sqrt[1 - Sin[j1]^2];
         csi2 = Sqrt[1 - (w s)^2 Sin[j1]^2];
         csj2 = Sqrt[1 - s^2 Sin[j1]^2];
         a = \[Rho]2 (1 - 2 \[Beta]2^2 p^2) - \[Rho]1 (1 - 2 \[Beta]1^2 p^2);
         b = \[Rho]2 (1 - 2 \[Beta]2^2 p^2) + 2 \[Rho]1 \[Beta]1^2 p^2;
         c = \[Rho]1 (1 - 2 \[Beta]1^2 p^2) + 2 \[Rho]2 \[Beta]2^2 p^2;
         d = 2 (\[Rho]2 \[Beta]2^2 - \[Rho]1 \[Beta]1^2);
         e = b csi1/\[Alpha]1 + c csi2/\[Alpha]2;
         f = b csj1/\[Beta]1 + c csj2/\[Beta]2;
         g = a - d csi1 csj2/(\[Alpha]1 \[Beta]2);
         h = a - d csi2 csj1/(\[Alpha]2 \[Beta]1);
         dd = e f + g h p^2;
         
         SdPu = -2 csj1 (a b + c d csi2 csj2/(\[Alpha]2 \[Beta]2)) p/(\[Alpha]1 dd);
         SdPu = Cancel[SdPu];
         SdPu = SdPu /. p -> Sin[j1]/\[Beta]1;
         SdPu = Cancel[SdPu];
         SdPu = SdPu /. j1 -> \[Gamma] \[Degree];
         
         SdSu = ((-b csj1/\[Beta]1 + c csj2/\[Beta]2) e + g p^2 (a + d csi2 csj1/(\[Alpha]2 \[Beta]1)))/dd;
         SdSu = Cancel[SdSu];
         SdSu = SdSu /. p -> Sin[j1]/\[Beta]1;
         SdSu = Cancel[SdSu];
         SdSu = SdSu /. j1 -> \[Gamma] \[Degree];
         
         SdPd = -2 \[Rho]1 csj1 g p/(\[Alpha]2 dd);
         SdPd = Cancel[SdPd];
         SdPd = SdPd /. p -> Sin[j1]/\[Beta]1;
         SdPd = Cancel[SdPd];
         SdPd = SdPd /. j1 -> \[Gamma] \[Degree];
         
         SdSd = 2 \[Rho]1 csj1 e/(\[Beta]2 dd);
         SdSd = Cancel[SdSd];
         SdSd = SdSd /. p -> Sin[j1]/\[Beta]1;
         SdSd = Cancel[SdSd];
         SdSd = SdSd /. j1 -> \[Gamma] \[Degree];
         
         (*Pu expressions*)
         csi1 = Sqrt[1 - (v/(w s))^2 Sin[i2]^2];
         csj1 = Sqrt[1 - (1/(w s))^2 Sin[i2]^2];
         csi2 = Sqrt[1 - Sin[i2]^2];
         csj2 = Sqrt[1 - (1/w)^2 Sin[i2]^2];
         a = \[Rho]2 (1 - 2 \[Beta]2^2 p^2) - \[Rho]1 (1 - 2 \[Beta]1^2 p^2);
         b = \[Rho]2 (1 - 2 \[Beta]2^2 p^2) + 2 \[Rho]1 \[Beta]1^2 p^2;
         c = \[Rho]1 (1 - 2 \[Beta]1^2 p^2) + 2 \[Rho]2 \[Beta]2^2 p^2;
         d = 2 (\[Rho]2 \[Beta]2^2 - \[Rho]1 \[Beta]1^2);
         e = b csi1/\[Alpha]1 + c csi2/\[Alpha]2;
         f = b csj1/\[Beta]1 + c csj2/\[Beta]2;
         g = a - d csi1 csj2/(\[Alpha]1 \[Beta]2);
         h = a - d csi2 csj1/(\[Alpha]2 \[Beta]1);
         dd = e f + g h p^2;
         
         PuPu = 2 \[Rho]2 csi2 f/(\[Alpha]1 dd);
         PuPu = Cancel[PuPu];
         PuPu = PuPu /. p -> Sin[i2]/\[Alpha]2;
         PuPu = Cancel[PuPu];
         PuPu = PuPu /. i2 -> \[Gamma] \[Degree];
         
         PuSu = -2 \[Rho]2 csi2 g p/(\[Beta]1 dd);
         PuSu = Cancel[PuSu];
         PuSu = PuSu /. p -> Sin[i2]/\[Alpha]2;
         PuSu = Cancel[PuSu];
         PuSu = PuSu /. i2 -> \[Gamma] \[Degree];
         
         PuPd = ((-b csi1/\[Alpha]1 + c csi2/\[Alpha]2) f - g p^2 (a + d csi2 csj1/(\[Alpha]2 \[Beta]1)))/dd;
         PuPd = Cancel[PuPd];
         PuPd = PuPd /. p -> Sin[i2]/\[Alpha]2;
         PuPd = Cancel[PuPd];
         PuPd = PuPd /. i2 -> \[Gamma] \[Degree];
         
         PuSd = 2 csi2 (a c + b d csi1 csj1/(\[Alpha]1 \[Beta]1)) p/(\[Beta]2 dd);
         PuSd = Cancel[PuSd];
         PuSd = PuSd /. p -> Sin[i2]/\[Alpha]2;
         PuSd = Cancel[PuSd];
         PuSd = PuSd /. i2 -> \[Gamma] \[Degree];
         
         (*Su expressions*)
         csi1 = Sqrt[1 - (v/s)^2 Sin[j2]^2];
         csj1 = Sqrt[1 - (1/s)^2 Sin[j2]^2];
         csi2 = Sqrt[1 - w^2 Sin[j2]^2];
         csj2 = Sqrt[1 - Sin[j2]^2];
         a = \[Rho]2 (1 - 2 \[Beta]2^2 p^2) - \[Rho]1 (1 - 2 \[Beta]1^2 p^2);
         b = \[Rho]2 (1 - 2 \[Beta]2^2 p^2) + 2 \[Rho]1 \[Beta]1^2 p^2;
         c = \[Rho]1 (1 - 2 \[Beta]1^2 p^2) + 2 \[Rho]2 \[Beta]2^2 p^2;
         d = 2 (\[Rho]2 \[Beta]2^2 - \[Rho]1 \[Beta]1^2);
         e = b csi1/\[Alpha]1 + c csi2/\[Alpha]2;
         f = b csj1/\[Beta]1 + c csj2/\[Beta]2;
         g = a - d csi1 csj2/(\[Alpha]1 \[Beta]2);
         h = a - d csi2 csj1/(\[Alpha]2 \[Beta]1);
         dd = e f + g h p^2;
         
         SuPu = 2 \[Rho]2 csj2 h p/(\[Alpha]1 dd);
         SuPu = Cancel[SuPu];
         SuPu = SuPu /. p -> Sin[j2]/\[Beta]2;
         SuPu = Cancel[SuPu];
         SuPu = SuPu /. j2 -> \[Gamma] \[Degree];
         
         SuSu = 2 \[Rho]2 csj2 e/(\[Beta]1 dd);
         SuSu = Cancel[SuSu];
         SuSu = SuSu /. p -> Sin[j2]/\[Beta]2;
         SuSu = Cancel[SuSu];
         SuSu = SuSu /. j2 -> \[Gamma] \[Degree];
         
         SuPd = 2 csj2 (a c + b d csi1 csj1/(\[Alpha]1 \[Beta]1)) p/(\[Alpha]2 dd);
         SuPd = Cancel[SuPd];
         SuPd = SuPd /. p -> Sin[j2]/\[Beta]2;
         SuPd = Cancel[SuPd];
         SuPd = SuPd /. j2 -> \[Gamma] \[Degree];
         
         SuSd = ((b csj1/\[Beta]1 - c csj2/\[Beta]2) e + h p^2 (a + d csi1 csj2/(\[Alpha]1 \[Beta]2)))/dd;
         SuSd = Cancel[SuSd];
         SuSd = SuSd /. p -> Sin[j2]/\[Beta]2;
         SuSd = Cancel[SuSd];
         SuSd = SuSd /. j2 -> \[Gamma] \[Degree];
         
         (*Return the 4x4 matrix of coefficients.*)
         {{PdPu, PdSu, PdPd, PdSd}, {SdPu, SdSu, SdPd, SdSd}, {PuPu, PuSu, PuPd, PuSd}, {SuPu, SuSu, SuPd, SuSd}}
         ];



(*
P-SV fluxes
This cell defines the vertical energy fluxes of plane P-SV waves in terms of layer parameter ratios: r for density ratio of layer 2 over layer 1,  s for shear velocity ratio of layer 2 over layer 1, v for ratio of compressional/shear velocity in the top layer, and w for similar ratio in the bottom layer.  The expressions should be evaluated for specific incidence angles \[Gamma] (degrees) and for specific v, w, r, s.  The starting point are the expressions in Aki & Richards (eq. 5.40).  The additional factors to convert these expressions to energy are developed here.  The waves are incident on the plane boundary of two homogeneous half-spaces where subscripts 1 and 2 refer to the upper and lower media.  The notebook creates sixteen expressions for flux energies of the sixteen scattered waves from the 4 possible incident waves (Pd,Sd,Pu,Su).  For instance, PdPu is the scattered Pu from an incident Pd.  After making this set of definitions, they are returned in a matrix (list of lists) like:   

{
{ePdPu,ePdSu,ePdPd,ePdSd},
{eSdPu,eSdSu,eSdPd,eSdSd},
{ePuPu,ePuSu,ePuPd,ePuSd},
{eSuPu,eSuSu,eSuPd,eSuSd}
}
*)

fluxPSV[] := Block[{coef, r, s, v, w, \[Alpha]1, \[Alpha]2, \[Beta]1, \[Beta]2, \[Rho]1, \[Rho]2, i1, i2, j1, j2, csi1, csi2, csj1, csj2, factor, \[Gamma]},
         
         (*Define the coefficients.*)
         coef = coefPSV[];
         
         (*Set up substitutions such that final expressions are in terms of v, w, r, s.*)
         \[Alpha]1 = v \[Beta]1;(*This eliminates \[Alpha]1 from the equations.*)
         \[Alpha]2 = w \[Beta]2;(*This eliminates \[Alpha]2 from the equations.*)
         \[Rho]2 = r \[Rho]1;(*This eliminates both densities as \[Rho]1 eventually cancels everywhere.*)
         \[Beta]2 = s \[Beta]1;(*This eliminates both velocities as \[Beta]1 eventually cancels everywhere.*)
         
         (*Pd expressions*)
         csi1 = Sqrt[1 - Sin[i1]^2];
         csj1 = Sqrt[1 - (1/v^2) Sin[i1]^2];
         csi2 = Sqrt[1 - (w^2/v^2 ) s^2 Sin[i1]^2];
         csj2 = Sqrt[1 - (1/v^2) s^2 Sin[i1]^2];
         
         ePdPu = Abs[PdPu]^2;
         ePdPu = Cancel[ePdPu];
         ePdPu = ePdPu /. i1 -> \[Gamma] \[Degree];
         
         factor = ((\[Beta]1 csj1)/(\[Alpha]1 csi1));
         factor = Cancel[factor];
         ePdSu = Abs[PdSu]^2 factor;
         ePdSu = Cancel[ePdSu];
         ePdSu = ePdSu /. i1 -> \[Gamma] \[Degree];
         
         factor = ((\[Rho]2 \[Alpha]2 csi2)/(\[Rho]1 \[Alpha]1 csi1));
         factor = Cancel[factor];
         ePdPd = Abs[PdPd]^2 factor;
         ePdPd = Cancel[ePdPd];
         ePdPd = ePdPd /. i1 -> \[Gamma] \[Degree];
         
         factor = ((\[Rho]2 \[Beta]2 csj2)/(\[Rho]1 \[Alpha]1 csi1));
         factor = Cancel[factor];
         ePdSd = Abs[PdSd]^2 factor;
         ePdSd = Cancel[ePdSd];
         ePdSd = ePdSd /. i1 -> \[Gamma] \[Degree];
         
         (*Sd expressions*)
         csi1 = Sqrt[1 - (\[Alpha]1/\[Beta]1)^2 Sin[j1]^2];
         csj1 = Sqrt[1 - Sin[j1]^2];
         csi2 = Sqrt[1 - (\[Alpha]2/\[Beta]1)^2 Sin[j1]^2];
         csj2 = Sqrt[1 - (\[Beta]2/\[Beta]1)^2 Sin[j1]^2];
         
         factor = ((\[Alpha]1 csi1)/(\[Beta]1 csj1));
         factor = Cancel[factor];
         eSdPu = Abs[SdPu]^2 factor;
         eSdPu = Cancel[eSdPu];
         eSdPu = eSdPu /. j1 -> \[Gamma] \[Degree];
         
         eSdSu = Abs[SdSu]^2;
         eSdSu = Cancel[eSdSu];
         eSdSu = eSdSu /. j1 -> \[Gamma] \[Degree];
         
         factor = ((\[Rho]2 \[Alpha]2 csi2)/(\[Rho]1 \[Beta]1 csj1));
         factor = Cancel[factor];
         eSdPd = Abs[SdPd]^2 factor;
         eSdPd = Cancel[eSdPd];
         eSdPd = eSdPd /. j1 -> \[Gamma] \[Degree];
         
         factor = ((\[Rho]2 \[Beta]2 csj2)/(\[Rho]1 \[Beta]1 csj1));
         factor = Cancel[factor];
         eSdSd = Abs[SdSd]^2 factor;
         eSdSd = Cancel[eSdSd];
         eSdSd = eSdSd /. j1 -> \[Gamma] \[Degree];
         
         (*Pu expressions*)
         csi1 = Sqrt[1 - (\[Alpha]1/\[Alpha]2)^2 Sin[i2]^2];
         csj1 = Sqrt[1 - (\[Beta]1/\[Alpha]2)^2 Sin[i2]^2];
         csi2 = Sqrt[1 - Sin[i2]^2];
         csj2 = Sqrt[1 - (\[Beta]2/\[Alpha]2)^2 Sin[i2]^2];
         
         factor = ((\[Rho]1 \[Alpha]1 csi1)/(\[Rho]2 \[Alpha]2 csi2));
         factor = Cancel[factor];
         ePuPu = Abs[PuPu]^2 factor;
         ePuPu = Cancel[ePuPu];
         ePuPu = ePuPu /. i2 -> \[Gamma] \[Degree];
         
         factor = ((\[Rho]1 \[Beta]1 csj1)/(\[Rho]2 \[Alpha]2 csi2));
         factor = Cancel[factor];
         ePuSu = Abs[PuSu]^2 factor;
         ePuSu = Cancel[ePuSu];
         ePuSu = ePuSu /. i2 -> \[Gamma] \[Degree];
         
         ePuPd = Abs[PuPd]^2;
         ePuPd = Cancel[ePuPd];
         ePuPd = ePuPd /. i2 -> \[Gamma] \[Degree];
         
         factor = ((\[Beta]2 csj2)/(\[Alpha]2 csi2));
         factor = Cancel[factor];
         ePuSd = Abs[PuSd]^2 factor;
         ePuSd = Cancel[ePuSd];
         ePuSd = ePuSd /. i2 -> \[Gamma] \[Degree];
         
         (*Su expressions*)
         csi1 = Sqrt[1 - (\[Alpha]1/\[Beta]2)^2 Sin[j2]^2];
         csj1 = Sqrt[1 - (\[Beta]1/\[Beta]2)^2 Sin[j2]^2];
         csi2 = Sqrt[1 - (\[Alpha]2/\[Beta]2)^2 Sin[j2]^2];
         csj2 = Sqrt[1 - Sin[j2]^2];
         
         factor = ((\[Rho]1 \[Alpha]1 csi1)/(\[Rho]2 \[Beta]2 csj2));
         factor = Cancel[factor];
         eSuPu = Abs[SuPu]^2 factor;
         eSuPu = Cancel[eSuPu];
         eSuPu = eSuPu /. j2 -> \[Gamma] \[Degree];
         
         factor = ((\[Rho]1 \[Beta]1 csj1)/(\[Rho]2 \[Beta]2 csj2));
         factor = Cancel[factor];
         eSuSu = Abs[SuSu]^2 factor;
         eSuSu = Cancel[eSuSu];
         eSuSu = eSuSu /. j2 -> \[Gamma] \[Degree];
         
         factor = ((\[Alpha]2 csi2)/(\[Beta]2 csj2));
         factor = Cancel[factor];
         eSuPd = Abs[SuPd]^2 factor;
         eSuPd = Cancel[eSuPd];
         eSuPd = eSuPd /. j2 -> \[Gamma] \[Degree];
         
         eSuSd = Abs[SuSd]^2;
         eSuSd = Cancel[eSuSd];
         eSuSd = eSuSd /. j2 -> \[Gamma] \[Degree];
         
         (*Return the 4x4 matrix of fluxes.*)
         {{ePdPu, ePdSu, ePdPd, ePdSd}, {eSdPu, eSdSu, eSdPd, eSdSd}, {ePuPu, ePuSu, ePuPd, ePuSd}, {eSuPu, eSuSu, eSuPd, eSuSd}}
         ];


(*
P-SV coefficients -- vacuum over elastic half space:
This cell defines the reflection and transmission coefficients of plane P-SV waves which are incident upwards on a boundary between an elastic half space and a vacuum.  The expressions should be evaluated for specific incidence angles \[Gamma] (degrees).  The starting point are the expressions in Aki & Richards (eqs. 5.27, 5.28, 5.31, 5.32).  The notebook creates 4 expressions for calculation of reflection/transmission coefficients of the 2 scattered waves from the 2 possible incident waves (Pu, Su).  For instance, PuPd is the scattered Pd from an incident Pu.  After making this set of definitions, they are returned in a matrix (list of lists) like:   

{
{PuPd,PuSd},
{SuPd,SuSd}
}
*)

coefPSVfs[] := Block[{u, p, f, h, \[Alpha], \[Beta], csi, csj, sni, snj, \[Gamma]},
         
         csi = Sqrt[1 - sni sni]; 
         csj = Sqrt[1 - snj snj]; 
         
         (*Set up substitution such that final expressions are in terms of u.*)
         \[Alpha] = u \[Beta]; 
         
         f = \[Beta]^(-2) - 2 p^2; 
         h = 4 p^2 ((csi csj)/(\[Alpha] \[Beta])); 
         
         PuPd = (h - f^2)/(f^2 + h) /. p -> sni/\[Alpha] /. snj -> (1/u) sni; 
         PuPd = Cancel[PuPd]; 
         PuPd = PuPd /. sni -> Sin[\[Gamma] \[Degree]]; 
         
         PuSd = 4 (csi/\[Alpha]) (sni/\[Beta]) (f/(f^2 + h)) /. p -> sni/\[Alpha] /. snj -> (1/u) sni; 
         PuSd = Cancel[PuSd]; 
         PuSd = PuSd /. sni -> Sin[\[Gamma] \[Degree]]; 
         
         SuPd = 4 (csj/\[Alpha]) (snj/\[Beta]) (f/(f^2 + h)) /. p -> snj/\[Beta] /. sni -> u snj; 
         SuPd = Cancel[SuPd]; 
         SuPd = SuPd /. snj -> Sin[\[Gamma] \[Degree]]; 
         
         SuSd = (f^2 - h)/(f^2 + h) /. p -> snj/\[Beta] /. sni -> u snj; 
         SuSd = Cancel[SuSd]; 
         SuSd = SuSd /. snj -> Sin[\[Gamma] \[Degree]]; 
         
         (*Return the 2x2 matrix of coefficients.*)
         {{PuPd, PuSd}, {SuPd, SuSd}}
         ];


(*
SH coefficients:
This cell defines the reflection and transmission coefficients of plane SH waves in terms of layer parameter ratios: r for density ratio of layer 2 over layer 1 and s for shear velocity ratio of layer 2 over layer 1.  The expressions should be evaluated for specific incidence angles \[Gamma] (degrees) and for specific r and s.  The starting point are the expressions in Aki & Richards (eq. 5.33).  The waves are incident on the plane boundary of two elastic, homogeneous half spaces where subscripts 1 and 2 refer to the upper and lower media.  The notebook creates 4 expressions for calculation of reflection/transmission coefficients of the 4 scattered waves from the 2 possible incident waves (Sd, Su).  For instance, SdSu is the scattered Su from an incident Sd.  After making this set of definitions, they are returned in a matrix (list of lists) like:   

{
{SdSu,SdSd},
{SuSu,SuSd}
}
*)

coefSH[] := Block[{r, s, \[Beta]1, \[Beta]2, \[Rho]1, \[Rho]2, j1, j2, csj1, csj2, \[Gamma], \[CapitalDelta]},
         
         (*Set up substitutions such that final expressions are in terms of r, s.*)
         \[Rho]2 = r \[Rho]1;(*This eliminates both densities as \[Rho]1 eventually cancels everywhere.*)
         \[Beta]2 = s \[Beta]1;(*This eliminates both velocities as \[Beta]1 eventually cancels everywhere.*)
         
         \[CapitalDelta] = \[Rho]1 \[Beta]1 csj1 + \[Rho]2 \[Beta]2 csj2;
   
         (*Sd expressions*)
         csj1 = Sqrt[1 - Sin[j1]^2];
         csj2 = Sqrt[1 - (\[Beta]2/\[Beta]1)^2 Sin[j1]^2];
         
         SdSu = (\[Rho]1 \[Beta]1 csj1 - \[Rho]2 \[Beta]2 csj2)/\[CapitalDelta];
         SdSu = Cancel[SdSu];
         SdSu = SdSu /. j1 -> \[Gamma] \[Degree];
         
         SdSd = (2 \[Rho]1 \[Beta]1 csj1)/\[CapitalDelta];
         SdSd = Cancel[SdSd];
         SdSd = SdSd /. j1 -> \[Gamma] \[Degree];
         
         (*Su expressions*)
         csj1 = Sqrt[1 - (\[Beta]1/\[Beta]2)^2 Sin[j2]^2];
         csj2 = Sqrt[1 - Sin[j2]^2];
               
         SuSu = (2 \[Rho]2 \[Beta]2 csj2)/\[CapitalDelta];
         SuSu = Cancel[SuSu];
         SuSu = SuSu /. j2 -> \[Gamma] \[Degree];
         
         SuSd = (-\[Rho]1 \[Beta]1 csj1 + \[Rho]2 \[Beta]2 csj2)/\[CapitalDelta];
         SuSd = Cancel[SuSd];
         SuSd = SuSd /. j2 -> \[Gamma] \[Degree];
         
         (*Return the 2x2 matrix of coefficients.*)
         {{SdSu, SdSd}, {SuSu, SuSd}}
         ];



(*
SH fluxes:
This cell defines the energy fluxes of plane SH waves in terms of layer parameter ratios: r for density ratio of layer 2 over layer 1 and s for shear velocity ratio of layer 2 over layer 1.  The expressions should be evaluated for specific incidence angles \[Gamma] (degrees) and for specific  r and s.  The waves are incident on the plane boundary of two homogeneous half-spaces where subscripts 1 and 2 refer to the upper and lower media.  The notebook creates 4 expressions for calculation of energy fluxes of the 4 scattered waves from the 2 possible incident waves (Sd, Su).  For instance, SdSu is the scattered Su from an incident Sd.  After making this set of definitions, they are returned in a matrix (list of lists) like:   

{
{eSdSu,eSdSd},
{eSuSu,eSuSd}
}
*)

fluxSH[] := Block[{coef, r, s, \[Beta]1, \[Beta]2, \[Rho]1, \[Rho]2, j1, j2, csj1, csj2, factor, \[Gamma], \[CapitalDelta]},
         
         (*Define the coefficients.*)
         coef = coefSH[];
         
         (*Set up substitutions such that final expressions are in terms of r, s.*)
         \[Rho]2 = r \[Rho]1;(*This eliminates both densities as \[Rho]1 eventually cancels everywhere.*)
         \[Beta]2 = s \[Beta]1;(*This eliminates both velocities as \[Beta]1 eventually cancels everywhere.*)
         
         (*Sd expressions*)
         csj1 = Sqrt[1 - Sin[j1]^2];
         csj2 = Sqrt[1 - (\[Beta]2/\[Beta]1)^2 Sin[j1]^2];
         
         eSdSu = Abs[SdSu]^2;
         eSdSu = Cancel[eSdSu];
         eSdSu = eSdSu /. j1 -> \[Gamma] \[Degree];
         
         factor = ((\[Rho]2 \[Beta]2 csj2)/(\[Rho]1 \[Beta]1 csj1)) /. \[Beta]2 -> s \[Beta]1;
         factor = Cancel[factor];
         eSdSd = factor Abs[SdSd]^2;
         eSdSd = Cancel[eSdSd];
         eSdSd = eSdSd /. j1 -> \[Gamma] \[Degree];
         
         (*Su expressions*)
         csj1 = Sqrt[1 - (\[Beta]1/\[Beta]2)^2 Sin[j2]^2];
         csj2 = Sqrt[1 - Sin[j2]^2];
         
         factor = ((\[Rho]1 \[Beta]1 csj1)/(\[Rho]2 \[Beta]2 csj2)) /. \[Beta]2 -> s \[Beta]1;
         factor = Cancel[factor];
         eSuSu = factor Abs[SuSu]^2;
         eSuSu = Cancel[eSuSu];
         eSuSu = eSuSu /. j2 -> \[Gamma] \[Degree];
         
         eSuSd = Abs[SuSd]^2;
         eSuSd = Cancel[eSuSd];
         eSuSd = eSuSd /. j2 -> \[Gamma] \[Degree];
         
         (*Return the 2x2 matrix of fluxes.*)
         {{eSdSu, eSdSd}, {eSuSu, eSuSd}}
         ];



End[]
EndPackage[]
